(ns resourceware.core
  "The core resourceware."
  (:require [cemerick.friend :as friend]
            [awohletz.validations :as v]
            [liberator.core]))

;Combine operation ----------------------------------------------------------------------

;Documenting the failure value of decisions.
;If a decision fails, we don't want to run any upstream decisions
;If it succeeds, we can run upstream decisions
(def liberator-decision-failure-results
  {:allowed?                   false
   :authorized?                false
   :charset-available?         false
   :can-post-to-gone?          false
   :can-post-to-missing?       false
   :can-put-to-missing?D       false
   :conflict?                  true
   :delete-enacted?            false
   :encoding-available?        false
   :etag-matches-for-if-match? false
   :etag-matches-for-if-none?  false
   :existed?                   false
   :exists?                    false
   :known-content-type?        false
   :known-method?              false
   :language-available?        false
   :malformed?                 true
   :media-type-available?      false
   :method-allowed?            false
   :modified-since?            false
   :moved-permanently?         true
   :moved-temporarily?         true
   :multiple-representations?  true
   :new?                       false
   :post-redirect?             true
   :put-to-different-url?      true
   :respond-with-entity?       true
   :service-available?         false
   :unmodified-since?          false
   :uri-too-long?              true
   :valid-content-header?      false
   :valid-entity-length?       false})

(defn liberator-failure-result [key]
  (get liberator-decision-failure-results key))

(defn liberator-decision? [key]
  (contains? liberator-decision-failure-results key))

(defn func?
  [f]
  (or (fn? f) (keyword? f)))

(defn ensure-function
  "Make sure x is a function that we can call.
  If x is a fn already, just return it. If it is a value, return a function that returns the value"
  [x]
  (if (func? x) x (constantly x)))

(defn merge-resources-with
  "Returns a map that consists of the rest of the maps conj-ed onto
  the first.  If a key occurs in more than one map, the mapping(s)
  from the latter (left-to-right) will be combined with the mapping in
  the result by calling (f key val-in-result val-in-latter)."
  [f & maps]
  (when (some identity maps)
    (let [merge-entry (fn [m e]
                        (let [k (key e) v (val e)]
                          (if (contains? m k)
                            (assoc m k (f k (get m k) v))
                            (assoc m k v))))
          merge2 (fn [m1 m2]
                   (reduce merge-entry (or m1 {}) (seq m2)))]
      (reduce merge2 maps))))

(defn combine-fns
  "Functions will be combined such that if the first func returns
  a value indicating success for a decision it will call the second func
  and merge the maps that the first and second funcs add to the context. If the
  first func returns a value indicating failure, it will return that value and
  not call the second func. If the element in the first map is a func but the second
  is of some other type, it will wrap the second value in a func that simply returns that value."
  [key curr newval]
  (let [false-result (liberator-failure-result key)
        next-fn (ensure-function newval)]
    (fn [context]
      (let [decision (curr context)
            ;if test returned a vector, take the first item of that vector as the result. Else we'll use the raw value
            ;returned by the test as the indication of success (truthy/falsy)
            result (if (vector? decision) (first decision) decision)
            ;if test returned a vector, use the second item of the vector as an update to the context
            ;else the return value itself represents an update to the context
            context-update (if (vector? decision) (second decision) decision)
            new-context (liberator.core/update-context context context-update)]
        (if (= false-result result)
          decision
          (next-fn new-context))))))

;This function is based on the one in Liberator
(defn resource-combine
  "Merge two values such that two maps a merged, two lists, two
  vectors and two sets are concatenated.

  Maps will be merged with maps. The map values will be merged
  recursively with this function.

  Lists, Vectors and Sets will be concatenated with values that are
  `coll?` and will preserve their type.

  Functions will be combined with functions or other types with combine-fns, but
  only if the function is registered for a decision point.

  For other combination of types the new value will be returned.

  If the newval has the metadata attribute `:replace` then it will
  replace the value regardless of the type."
  [key curr newval]
  (cond
    (-> newval meta :replace) newval
    (and (map? curr) (map? newval)) (merge-resources-with resource-combine curr newval)
    (and (list? curr) (coll? newval)) (concat curr newval)
    (and (vector? curr) (coll? newval)) (concat curr newval)
    (and (set? curr) (coll? newval)) (set (concat curr newval))
    (and (liberator-decision? key) (func? curr)) (combine-fns key curr newval)
    :otherwise newval))

(defn combine [resource new-resource]
  (merge-resources-with resource-combine resource new-resource))

;Utils ----------------------------------------------------------------------

(defn new-resource
  "The preferred way to create a new resource (which is just a hashmap). You can either
  specify one media type which will be used for everything, media-type, or you can
  give the following:

  available-media-types = list of media types this resource can negotiate on.

  media-type-for-errors = the media type to use when an error occurs before the available
  media types are processed."
  ([media-type]
    (new-resource [media-type] media-type))
  ([available-media-types media-type-for-errors]
   {:pre [available-media-types media-type-for-errors]}
   {:available-media-types available-media-types
    ::media-type           media-type-for-errors}))

;Basic middleware ----------------------------------------------------------------------

(defn wrap-allowed
  "Resource middleware to add a custom 'allowed?' func to the resource.
  This func must take the context, a success fn, and a failure fn.
  It must return the value that success fn or failure fn return.
  It must pass a string reason to failure fn.
  success fn takes no arguments."
  [resource allowed?]
  (combine resource
           {:allowed? (fn [ctx]
                        (allowed? ctx
                                  (fn []
                                    true)
                                  (fn [reason]
                                    [false {::reason reason}])))}))

(defn wrap-allowed-methods
  "Resource middleware to add allowed methods to the resource. E.g. :get, :post, :options, ..."
  [resource allowed-methods]
  (combine resource
           {:allowed-methods allowed-methods}))

(defn wrap-etag
  "Resource middleware to add an ETag function"
  [resource etag]
  (combine resource
           {:etag etag}))


;Authorization middleware ----------------------------------------------------------------------

(defn decision?
  "Returns whether auth-fn returned true or false"
  [retn]
  (if (vector? retn)
    (first retn)
    retn))

(defn added-to-ctx
  "See if the response from auth-fn gave anything to add to the context"
  [retn]
  (if (vector? retn)
    (second retn)
    (if (map? retn)
      retn
      {})))

(defn decision-fn [decide media-type failure-reason]
  (fn [ctx]
    (let [retn (decide ctx)]
      (vector (decision? retn) (merge {:representation {:media-type media-type}}
                                      {::reason failure-reason}
                                      (added-to-ctx retn))))))

(def default-unauthenticated-reason "You are not authenticated to access that resource.")
(def default-unauthorized-reason "You are not authorized to access that resource.")

(defn wrap-unauthenticated-response [resource handler]
  (combine resource {:handle-unauthorized handler}))

(defn wrap-unauthorized-response [resource handler]
  (combine resource {:handle-forbidden handler}))

(defn wrap-auth*
  ([resource authenticated? authorized? err-handler]
   (let [err-handler (fn [{:keys [::reason]}] (err-handler reason))]
     (wrap-auth* resource authenticated? authorized? err-handler err-handler)))
  ([{:keys [::media-type] :as resource} authenticated? authorized? authentication-error-handler
    authorization-error-handler]
   (-> resource
       (combine {:authorized? (decision-fn authenticated? media-type default-unauthenticated-reason)
                 :allowed?    (decision-fn authorized? media-type default-unauthorized-reason)})
       (wrap-unauthenticated-response authentication-error-handler)
       (wrap-unauthorized-response authorization-error-handler))))

(def friend-identity-present? (comp boolean friend/identity :request))

(defn wrap-authentication
  "Resource middleware to require Friend authentication for this resource.
  err-handler = function that accepts a string reason and produces a Liberator handler response"
  [resource err-handler]
  (wrap-auth* resource
              friend-identity-present?
              true
              err-handler))

(defn roles-with-methods
  "Returns an authorization function that checks if the authenticated
  user has the specified roles. (This is the usual friend behavior.)"
  [roles-meths]
  (fn [{:keys [request-method] :as req}]
    (let [role (friend/authorized? (keys roles-meths) req)
          authenticated? role
          authorized? (some #{request-method} (get roles-meths role))]
      [authenticated? {::authorized? authorized?}])))

(defn wrap-role-and-methods-auth
  "Resource middleware to require certain Friend roles to be present to access this resource.
  Every role has a list of methods that role can access on the resource.

  roles-meths = {role [:get :post ...]}
  err-handler = function that accepts a string reason and produces a Liberator handler response"
  [resource roles-meths err-handler]
  (wrap-auth* resource
              (comp (roles-with-methods roles-meths) :request)
              (fn [{:keys [::authorized?]}]
                authorized?)
              err-handler))

;Validation middleware ----------------------------------------------------------------------

(defn malformed-retn [reason media-type]
  [true
   (merge {:representation {:media-type media-type}}
          {::reason reason})])

(defn wrap-validation
  "Resource middleware to have a custom function validate the parameters.
  Validation-fn must take the context and return either [true, <map to merge with context>]
  (if the params are valid) or [false, <validation errors>]

  err-handler = function that accepts the validation errors and produces a Liberator handler response

  check-for-methods = set of methods (e.g. #{:get :post}) that the validations will run on. If
  the request method doesn't match a method in this list, then no validations will run and it will
  assume the request is valid. Defaults to #{:post}"
  ([resource validation-fn err-handler]
   (wrap-validation resource validation-fn err-handler #{:post}))
  ([{:keys [::media-type] :as resource} validation-fn err-handler check-for-methods]
   {:pre [(set? check-for-methods)
          media-type
          validation-fn
          err-handler]}
   (-> resource
       (wrap-allowed-methods (vec check-for-methods))
       (combine
             {:malformed?       (fn [ctx]
                                  (if (contains? check-for-methods (get-in ctx [:request :request-method]))
                                    (let [[valid? added-to-ctx] (validation-fn ctx)]
                                      (if (true? valid?)
                                        [false added-to-ctx]
                                        (malformed-retn added-to-ctx media-type)))))
              :handle-malformed (fn [{:keys [::reason] :as ctx}]
                                  (err-handler reason))}))))

;Creation middleware ----------------------------------------------------------------------

(defn wrap-creation
  "Resource middleware to have typical POST workflow for a resource.

  create! = a ValidatedFn that takes the context and causes whatever side-effects necessary (e.g. insert into the DB)
  and returns a URI to the newly-created resource.

  malformed-handler = function that accepts a list of validation errors -- {:status ... :reason ...} --
  and produces a Liberator handler response.

  created-handler = function that takes the URI of the new resource (string) and produces a Liberator handler response."
  ([resource create!]
   (wrap-creation resource create! identity identity))
  ([resource create! malformed-handler created-handler]
   (-> resource
       (wrap-validation
         (fn [{{:keys [request-method]} :request :as ctx}]
           (when (= request-method :post)
             (v/valid? create! ctx
                       (fn [f]
                         [true {::create-fn f}])
                       (fn [errors]
                         [false errors]))))
         malformed-handler)
       (wrap-allowed-methods [:post :options])
       (combine
         {:post-redirect? false
          :new?           true
          :post!          (fn [{:keys [::create-fn]}]
                            {::new-uri (create-fn)})
          :handle-created (fn [{:keys [::new-uri]}]
                            (created-handler new-uri))}))))

;Reading middleware ----------------------------------------------------------------------

(defn wrap-reading
  "Resource middleware to have a typical GET workflow for a resource.
  get = fn that takes the context and returns the response"
  [resource get]
  (combine resource
           {:allowed-methods [:get]
            :handle-ok       get}))


