(ns resourceware.hal
  "A layer above resourceware.core that adds Hypertext Application Layer functionality."
  (:require [resourceware.core :as r]
            [awohletz.hal-gen :as h]))

;Utilities ----------------------------------------------------------------------

(defn resources->links [resources]
  (map (fn [{:keys [uri rel]}]
         {rel uri})
       resources))

(defn resource-links
  "Create links to self and any resources given in links"
  [uri links]
  (h/links (merge {"self" uri}
                  (resources->links links))))

(defn err-handler-hal
  "Default error handler that returns a HAL response"
  [{:keys [uri]} reason links]
  (h/resource (resource-links uri links)
              {:reason reason}))

(defn err-handler-hal-fn
  "Create a configured error handler HAL function"
  [resource links]
  (fn [reason] (err-handler-hal resource reason links)))

(defn new-resource
  "The preferred way to create a new HAL-enabled resource (which is just a hashmap).
  Same args as resourceware.core/new-resource with one additional:

  uri = the URI of the resource."
  [available-media-types media-type-for-errors uri]
  {:pre [available-media-types media-type-for-errors uri]}
  (merge (r/new-resource available-media-types media-type-for-errors)
         {::uri uri}))

;Auth ware ----------------------------------------------------------------------

(defn wrap-authentication
  "Resource middleware that is the same as wrap-authentication except it produces a HAL response.
  links-auth-err = vector of other resources to link to if this middleware
  produces an authentication/authorization error response."
  [resource links-auth-err]
  (r/wrap-authentication
    resource
    (err-handler-hal-fn resource links-auth-err)))

(defn wrap-role-and-methods-auth
  "Resource middleware to require certain Friend roles to be present to access this resource.
  Every role has a list of methods that role can access on the resource.

  roles = {role [:get :post ...]}"
  [resource roles-meths links-auth-err]
  (r/wrap-role-and-methods-auth resource
                                roles-meths
                                (err-handler-hal-fn resource links-auth-err)))

;Validation ware ----------------------------------------------------------------------

(defn wrap-validation
  "Resource middleware to wrap validation and produce a HAL response
  links-malformed = a vector of other resources to link to when validation fails"
  [resource validation-fn links-malformed]
  (r/wrap-validation resource
                     validation-fn
                     (err-handler-hal-fn resource links-malformed)))

;Creation ----------------------------------------------------------------------

(defn wrap-creation
  "Resource middleware same as wrap-creation but produces HAL responses
  links-created = vector of other resources to link to upon created response.
  Will insert a link with 'new' rel to the new resource URI"
  [{:keys [::uri] :as resource} create! links-created links-malformed]
  (r/wrap-creation resource
                   create!
                   (err-handler-hal-fn resource links-malformed)
                   (fn [new-uri]
                     (h/resource (resource-links uri links-created)
                                 (h/link "new" new-uri)))))