(ns aws.iam.coercion
  (:require[aws.auth :as auth]
           [aws.client]
           [aws.coerce.to-clj :refer [->clj]]
           [aws.coerce.to-sdk :refer [->sdk]]
           [aws.util :as u])
  (:import [software.amazon.awssdk.core.client.config ClientOverrideConfiguration]
           [software.amazon.awssdk.http SdkHttpClient]
           [software.amazon.awssdk.http.apache ApacheHttpClient]
           [software.amazon.awssdk.regions Region]
           [software.amazon.awssdk.services.iam IamClient]
           [software.amazon.awssdk.services.iam.model
            ListUsersRequest
            ListUsersResponse
            User]))

;; ----
;; client
;;
;; ----

(defmethod ->sdk IamClient [_ iam-client]
  (let [{:keys [http-client
                credentials-provider
                endpoint-override
                override-configuration
                region]
         :or {credentials-provider (auth/default-credentials-provider)}} iam-client]
    (.build
     (doto (IamClient/builder)
       (.credentialsProvider credentials-provider)
       (cond-> http-client (.httpClient ^SdkHttpClient (->sdk ApacheHttpClient http-client)))
       (cond-> region (.region (->sdk Region region)))
       (cond-> endpoint-override (.endpointOverride endpoint-override))
       (cond-> override-configuration (.overrideConfiguration ^ClientOverrideConfiguration (->sdk ClientOverrideConfiguration override-configuration)))))))

;; ----
;; to clojure
;;
;; ----

(extend-type User
  aws.coerce.to-clj/ToClojure
  (to-clj [user]
    (u/only-valid-values
     {:arn (.arn user)
      :create-date (.createDate user)
      :password-last-used (.passwordLastUsed user)
      :path (.path user)
      :user-id (.userId user)
      :user-name (.userName user)})))

(extend-type ListUsersResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:users (mapv (fn [user](->clj user)) (.users response))
      :is-truncated (.isTruncated response)
      :marker (.marker response)})))

;; ----
;; to sdk
;;
;; ---

(defmethod ->sdk ListUsersRequest [_ list-users-request]
  (let [{:keys [marker
                max-items
                path-prefix]
         :or {max-items 100}} list-users-request]
    (.build
     (doto (ListUsersRequest/builder)
       (.maxItems (int max-items))
       (cond-> marker (.marker marker)
               path-prefix (.pathPrefix path-prefix))))))
