(ns aws.s3
  (:require [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.s3.coercion]
            [aws.s3.client :as client])
  (:import [software.amazon.awssdk.services.s3
            S3Client]
           [software.amazon.awssdk.core.sync
            RequestBody]
           [software.amazon.awssdk.services.s3.model
            DeleteObjectRequest
            GetObjectRequest
            GetObjectTaggingRequest
            HeadObjectRequest
            ListObjectsV2Request
            ListObjectsV2Response
            PutObjectRequest
            PutObjectTaggingRequest]))

;; TODO: make this return a lazy sequence of ALL buckets (see list-objects below)
(defn list-buckets [^S3Client client]
  (->clj (.listBuckets client)))

(defn list-objects
  "Lazily list all objects in a [bucket, prefix] s3 location
    even if there are more that 1000 objects in the location"

  ([client bucket prefix] (list-objects client {:bucket bucket :prefix prefix}))

  ([^S3Client client request]
   (let [list-fn (fn [req]
                   (->clj (.listObjectsV2 client ^ListObjectsV2Request (->sdk ListObjectsV2Request request))))

         next-batch-fn (fn [{:keys [truncated? next-continuation-token] :as object-listing}]
                         (when truncated?
                           (list-fn (assoc request :continuation-token next-continuation-token))))]

     (->>  (list-fn request)
           (iterate next-batch-fn)
           (take-while (complement nil?))
           (mapcat :contents)))))

(defn- put-object* [^S3Client client ^PutObjectRequest request ^RequestBody request-body]
  (->clj (.putObject client request request-body)))

(defn put-object

  ([client bucket key content]
   (put-object client {:bucket bucket :key key} content))

  ([^S3Client client bucket key input-stream length]
   (put-object*
    client
    (->sdk PutObjectRequest {:bucket bucket :key key})
    (->sdk RequestBody input-stream length)))

  ([^S3Client client request content]
   (put-object*
    client
    (->sdk PutObjectRequest request)
    (->sdk RequestBody content))))

(defn get-object-metadata
  ([client bucket key] (get-object-metadata client {:bucket bucket :key key}))
  ([^S3Client client request]
   (->clj (.headObject client ^HeadObjectRequest (->sdk HeadObjectRequest request)))))

;; -------
;;  NOTE: there is no round-tripping of tags that begin life with keywords as keys
;; (use walk/keywordize-keys on return trip if needed)
;;
;; -----

(defn get-object-tagging
  ([client bucket key] (get-object-tagging client {:bucket bucket :key key}))
  ([^S3Client client request]
   (->clj (.getObjectTagging client ^GetObjectTaggingRequest (->sdk GetObjectTaggingRequest request)))))

(defn put-object-tagging
  ([client bucket key tags] (put-object-tagging client {:bucket bucket :key key :tagging tags}))
  ([^S3Client client request]
   (->clj (.putObjectTagging client ^PutObjectTaggingRequest (->sdk PutObjectTaggingRequest request)))))

(defn get-object
  ([client bucket key] (get-object client {:bucket bucket :key key}))
  ([^S3Client client request]
   (->clj (.getObject client ^GetObjectRequest (->sdk GetObjectRequest request)))))

(defn delete-object
  ([client bucket key] (delete-object client {:bucket bucket :key key}))
  ([^S3Client client request]
   (->clj (.deleteObject client ^DeleteObjectRequest (->sdk DeleteObjectRequest request)))))


