(ns aws.s3.coercion
  (:require [aws.auth :as auth]
            [aws.client]
            [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.util :as u]
            [clojure.java.io :as io]
            [clojure.walk :as walk])
  (:import [software.amazon.awssdk.core
            ResponseInputStream]
           [software.amazon.awssdk.core.client.config ClientOverrideConfiguration]
           [software.amazon.awssdk.core.sync
            RequestBody]
           [software.amazon.awssdk.http SdkHttpClient]
           [software.amazon.awssdk.http.apache ApacheHttpClient]
           [software.amazon.awssdk.regions Region]
           [software.amazon.awssdk.services.s3 S3Client S3Configuration]
           [software.amazon.awssdk.services.s3.model
            Bucket
            DeleteObjectRequest
            DeleteObjectResponse
            GetObjectRequest
            GetObjectResponse
            GetObjectTaggingRequest
            GetObjectTaggingResponse
            HeadObjectRequest
            HeadObjectResponse
            ListBucketsResponse
            ListObjectsV2Request
            ListObjectsV2Response
            Owner
            PutObjectRequest
            PutObjectResponse
            PutObjectTaggingRequest
            PutObjectTaggingResponse
            S3Object
            Tag
            Tagging]))

;; ----
;; client
;;
;; ----

(defmethod ->sdk S3Configuration [_ s3-configuration]
  (let [{:keys [accelerate-mode-enabled
                dual-stack-enabled
                path-style-access-enabled]
         :or {accelerate-mode-enabled false
              dual-stack-enabled false
              path-style-access-enabled false}} s3-configuration]
    (.build
     (doto (S3Configuration/builder)
       (.accelerateModeEnabled accelerate-mode-enabled)
       (.dualstackEnabled dual-stack-enabled)
       (.pathStyleAccessEnabled path-style-access-enabled)))))

(defmethod ->sdk S3Client [_ s3-client]
  (let [{:keys [http-client
                service-configuration
                credentials-provider
                endpoint-override
                override-configuration
                region]
         :or {credentials-provider (auth/default-credentials-provider)}} s3-client]
    (.build
     (doto (S3Client/builder)
       (.credentialsProvider credentials-provider)
       (cond-> http-client (.httpClient ^SdkHttpClient (->sdk ApacheHttpClient http-client)))
       (cond-> region (.region (->sdk Region region)))
       (cond-> endpoint-override (.endpointOverride endpoint-override))
       (cond-> service-configuration (.serviceConfiguration ^S3Configuration (->sdk S3Configuration service-configuration)))
       (cond-> override-configuration (.overrideConfiguration ^ClientOverrideConfiguration (->sdk ClientOverrideConfiguration override-configuration)))))))

;; -------
;;  generating a RequestBody for put-object
;;
;; -----

(def byte-array-class (class (byte-array nil)))

(defn to-request-body-dispatch-fn [source & args]
  (class source))

(defmulti to-request-body #'to-request-body-dispatch-fn)

(defmethod to-request-body byte-array-class [#^bytes ba]
  (RequestBody/fromBytes ba))

(defmethod to-request-body java.nio.ByteBuffer [^java.nio.ByteBuffer bb]
  (RequestBody/fromByteBuffer bb))

(defmethod to-request-body java.nio.file.Path [^java.nio.file.Path p]
  (RequestBody/fromFile p))

(defmethod to-request-body java.io.File [^java.io.File f]
  (RequestBody/fromFile f))

(defmethod to-request-body java.io.InputStream [is content-length]
  (RequestBody/fromInputStream is content-length))

(defmethod to-request-body java.lang.String [^String s]
  (RequestBody/fromString s))

(defmethod to-request-body nil [_]
  (RequestBody/empty))


;; -------
;;  from simpledb sdk objects to clojure
;;
;; -----

(extend-type Bucket
  aws.coerce.to-clj/ToClojure
  (to-clj [bucket]
    {:creation-date (.creationDate bucket)
     :name (.name bucket)}))

(extend-type DeleteObjectResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:delete-marker (.deleteMarker response)
      :request-charged (.requestChargedAsString response)
      :version-id (.versionId response)})))

(extend-type GetObjectResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (let [metadata (.metadata response)]
      (u/only-valid-values
       {:accept-ranges (.acceptRanges response)
        :cache-control (.cacheControl response)
        :content-disposition (.contentDisposition response)
        :content-encoding (.contentEncoding response)
        :content-language (.contentLanguage response)
        :content-length (.contentLength response)
        :content-range (.contentRange response)
        :content-type (.contentType response)
        :delete-marker (.deleteMarker response)
        :etag (.eTag response)
        :expiration (.expiration response)
        :expires (.expires response)
        :last-modified (.lastModified response)
        :metadata (when metadata (into {} metadata))
        :missing-meta (.missingMeta response)
        :parts-count (.partsCount response)
        :replication-status (.replicationStatusAsString response)
        :request-charged (.requestChargedAsString response)
        :restore (.restore response)
        :server-side-encryption (.serverSideEncryptionAsString response)
        :sse-customer-algorithm (.sseCustomerAlgorithm response)
        :sse-customer-md5 (.sseCustomerKeyMD5 response)
        :sse-kms-key-id (.ssekmsKeyId response)
        :storage-class (.storageClassAsString response)
        :tag-count (.tagCount response)
        :version-id (.versionId response)
        :website-redirect-location (.websiteRedirectLocation response)}))))

(extend-type GetObjectTaggingResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:tagging (into {} (mapv (fn [tag] (->clj tag)) (.tagSet response)))
      :version-id (.versionId response)})))

(extend-type HeadObjectResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (let [metadata (.metadata response)]
      (u/only-valid-values
       {:accept-ranges (.acceptRanges response)
        :cache-control (.cacheControl response)
        :content-disposition (.contentDisposition response)
        :content-encoding (.contentEncoding response)
        :content-language (.contentLanguage response)
        :content-length (.contentLength response)
        :content-type (.contentType response)
        :delete-marker (.deleteMarker response)
        :etag (.eTag response)
        :expiration (.expiration response)
        :expires (.expires response)
        :last-modified (.lastModified response)
        :metadata (when metadata (into {} metadata))
        :missing-meta (.missingMeta response)
        :parts-count (.partsCount response)
        :replication-status (.replicationStatusAsString response)
        :request-charged (.requestChargedAsString response)
        :restore (.restore response)
        :server-side-encryption (.serverSideEncryptionAsString response)
        :sse-customer-algorithm (.sseCustomerAlgorithm response)
        :sse-customer-md5 (.sseCustomerKeyMD5 response)
        :sse-kms-key-id (.ssekmsKeyId response)
        :storage-class (.storageClassAsString response)
        :version-id (.versionId response)
        :website-redirect-location (.websiteRedirectLocation response)}))))

(extend-type ListBucketsResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    {:owner (->clj (.owner response))
     :buckets (mapv ->clj (.buckets response))}))

(extend-type ListObjectsV2Response
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (let [encoding-type (.encodingType response)
          bucket (.name response)]
      (u/only-valid-values
       {:contents (mapv (fn [s3-object] (with-meta (->clj s3-object) {:bucket bucket})) (.contents response))
        :continuation-token (.continuationToken response)
        :delimiter (.delimiter response)
        :encoding-type (when encoding-type (str encoding-type))
        :truncated? (.isTruncated response)
        :key-count (.keyCount response)
        :max-keys (.maxKeys response)
        :name bucket
        :next-continutation-token (.nextContinuationToken response)
        :prefix (.prefix response)
        :start-after (.startAfter response)}))))

(extend-type Owner
  aws.coerce.to-clj/ToClojure
  (to-clj [owner]
    (u/only-valid-values
     {:id (.id owner)
      :display-name (.displayName owner)})))

(extend-type PutObjectResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:etag (.eTag response)
      :expiration (.expiration response)
      :request-charged (.requestChargedAsString response)
      :server-side-encryption (.serverSideEncryptionAsString response)
      :sse-customer-algorithm (.sseCustomerAlgorithm response)
      :sse-customer-key-md5 (.sseCustomerKeyMD5 response)
      :sse-kms-key-id (.ssekmsKeyId response)
      :version-id (.versionId response)})))

(extend-type PutObjectTaggingResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:version-id (.versionId response)})))

(extend-type ResponseInputStream
  aws.coerce.to-clj/ToClojure
  (to-clj [input-stream]
    {:input-stream input-stream
     :response (->clj (.response input-stream))}))

(extend-type S3Object
  aws.coerce.to-clj/ToClojure
  (to-clj [object]
    (let [owner (.owner object)]
      (u/only-valid-values
       {:key (.key object)
        :etag (.eTag object)
        :last-modified (.lastModified object)
        :owner (when owner (->clj owner))
        :size (.size object)
        :storage-class (.storageClassAsString object)}))))

(extend-type Tag
  aws.coerce.to-clj/ToClojure
  (to-clj [tag]
    [(.key tag) (.value tag)]))

;; -------
;;  from clojure to simbpledb sdk objects
;;
;; -----

(defmethod ->sdk DeleteObjectRequest [_ delete-object-request]
  (let [{:keys [bucket
                key
                mfa
                request-payer
                version-id]} delete-object-request]
    (.build
     (doto (DeleteObjectRequest/builder)
       (.bucket bucket)
       (.key key)
       (cond-> mfa (.mfa mfa)
               request-payer (.requestPayer ^String request-payer)
               version-id (.versionId version-id))))))

(defmethod ->sdk GetObjectRequest [_ get-object-request]
  (let [{:keys [bucket
                key
                version-id]} get-object-request]
    (.build
     (doto (GetObjectRequest/builder)
       (.bucket bucket)
       (.key key)
       (cond-> version-id (.versionId version-id))))))

(defmethod ->sdk GetObjectTaggingRequest [_ get-object-tagging-request]
  (let [{:keys [bucket
                key
                version-id]} get-object-tagging-request]
    (.build
     (doto (GetObjectTaggingRequest/builder)
       (.bucket bucket)
       (.key key)
       (cond-> version-id (.versionId version-id))))))

(defmethod ->sdk HeadObjectRequest [_ head-object-request]
  (let [{:keys [bucket
                key
                version-id]} head-object-request]
    (.build
     (doto (HeadObjectRequest/builder)
       (.bucket bucket)
       (.key key)
       (cond-> version-id (.versionId version-id))))))

(defmethod ->sdk ListObjectsV2Request [_ list-object-v2-request]
  (let [{:keys [bucket
                prefix
                continuation-token
                delimeter
                encoding-type
                fetch-owner
                max-keys
                request-payer
                start-after]
         :or {max-keys 1000}} list-object-v2-request]
    (.build
     (doto (ListObjectsV2Request/builder)
       (.bucket bucket)
       (.prefix prefix)
       (.maxKeys (int max-keys))
       (cond-> continuation-token (.continuationToken  continuation-token)
               delimeter (.delimiter  delimeter)
               encoding-type (.encodingType ^String encoding-type)
               fetch-owner (.fetchOwner fetch-owner)
               request-payer (.requestPayer ^String request-payer)
               start-after (.startAfter start-after))))))

(defmethod ->sdk PutObjectRequest [_ put-object-request]
  (let [{:keys [bucket
                key
                content-type
                content-length
                metadata
                tagging]} put-object-request]
    (.build
     (doto (PutObjectRequest/builder)
       (.bucket bucket)
       (.key key)
       (cond-> content-type (.contentType content-type)
               content-length (.contentLength content-length)
               metadata (.metadata (walk/stringify-keys metadata))
               tagging (.tagging ^Tagging (->sdk Tagging tagging)))))))

(defmethod ->sdk PutObjectTaggingRequest [_ put-object-tagging-request]
  (let [{:keys [bucket
                key
                content-md5
                tagging
                version-id]} put-object-tagging-request]
    (.build
     (doto (PutObjectTaggingRequest/builder)
       (.bucket bucket)
       (.key key)
       (.tagging ^Tagging (->sdk Tagging tagging))
       (cond-> content-md5 (.contentMD5 content-md5)
               version-id (.versionId version-id))))))

(defmethod ->sdk RequestBody [_ & args]
  (apply to-request-body args))

(defmethod ->sdk Tag [_ entry]
  (let [[k v]  entry]
    (.build
     (doto (Tag/builder)
       (.key (name k))
       (.value (str v))))))

(defmethod ->sdk Tagging [_ tagging]
  (.build
   (doto (Tagging/builder)
     (.tagSet
      #^"[Lsoftware.amazon.awssdk.services.s3.model.Tag;"
      (into-array
       Tag
       (mapv
        (fn [entry]
          (->sdk Tag entry))
        tagging))))))
