(ns aws.sdb.coercion
  (:require [aws.auth :as auth]
            [aws.client]
            [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.util :as util])
  (:import [software.amazon.awssdk.core.client.config ClientOverrideConfiguration]
           [software.amazon.awssdk.http SdkHttpClient]
           [software.amazon.awssdk.http.apache ApacheHttpClient]
           [software.amazon.awssdk.regions Region]
           [software.amazon.awssdk.services.simpledb SimpleDbClient]
           [software.amazon.awssdk.services.simpledb.model
            Attribute
            CreateDomainRequest
            CreateDomainResponse
            DeleteAttributesRequest
            DeleteAttributesResponse
            DeleteDomainRequest
            DeleteDomainResponse
            DomainMetadataRequest
            DomainMetadataResponse
            GetAttributesRequest
            GetAttributesResponse
            Item
            ListDomainsRequest
            ListDomainsResponse
            PutAttributesRequest
            PutAttributesResponse
            ReplaceableAttribute
            SelectRequest
            SelectResponse
            UpdateCondition]))

;; ----
;; client
;;
;; -----

(defmethod ->sdk SimpleDbClient [_ simpledb-client]
  (let [{:keys [http-client
                credentials-provider
                endpoint-override
                override-configuration
                region]
         :or {credentials-provider (auth/default-credentials-provider)}} simpledb-client]
    (.build
     (doto (SimpleDbClient/builder)
       (.credentialsProvider credentials-provider)
       (cond-> http-client (.httpClient ^SdkHttpClient (->sdk ApacheHttpClient http-client)))
       (cond-> region (.region (->sdk Region region)))
       (cond-> endpoint-override (.endpointOverride endpoint-override))
       (cond-> override-configuration (.overrideConfiguration ^ClientOverrideConfiguration (->sdk ClientOverrideConfiguration override-configuration)))))))

;; -------
;;  to clojure
;;
;; -----

(extend-type Attribute
  aws.coerce.to-clj/ToClojure
  (to-clj [attribute]
    {:name (.name attribute)
     :value (.value attribute)}))

(extend-type CreateDomainResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [attribute]
    {}))

(extend-type DeleteAttributesResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [attribute]
    {}))

(extend-type DeleteDomainResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [attribute]
    {}))

(extend-type DomainMetadataResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [domain-metadata-response]
    (util/only-valid-values
     {:attribute-name-count (.attributeNameCount domain-metadata-response)
      :attritute-names-size-bytes (.attributeNamesSizeBytes domain-metadata-response)
      :attribute-value-count (.attributeValueCount domain-metadata-response)
      :attribute-values-size-bytes (.attributeValuesSizeBytes domain-metadata-response)
      :item-count (.itemCount domain-metadata-response)
      :item-names-size-bytes (.itemNamesSizeBytes domain-metadata-response)
      :timestamp (.timestamp domain-metadata-response)})))

(extend-type GetAttributesResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [get-attributes-response]
    (util/only-valid-values
     {:attributes (mapv ->clj (.attributes get-attributes-response))})))

(extend-type Item
  aws.coerce.to-clj/ToClojure
  (to-clj [item]
    {:attributes (mapv ->clj (.attributes item))
     :name (.name item)}))

(extend-type ListDomainsResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [list-domains-response]
    (util/only-valid-values
     {:domain-names (into [] (.domainNames list-domains-response))
      :next-token (.nextToken list-domains-response)})))

(extend-type PutAttributesResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [attribute]
    {}))

(extend-type SelectResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (util/only-valid-values
     {:items (mapv ->clj (.items response))
      :next-token (.nextToken response)})))

;; -------
;;  to sdk objects
;;
;; -----

(defmethod ->sdk Attribute [_ attribute]
  (let [{:keys [name
                value]} attribute]
    (.build
     (doto (Attribute/builder)
       (.name (clojure.core/name name))
       (.value value)))))

(defmethod ->sdk CreateDomainRequest [_ create-domain-request]
  (let [{:keys [domain-name]} create-domain-request]
    (.build
     (doto (CreateDomainRequest/builder)
       (.domainName (name domain-name))))))

(defmethod ->sdk DeleteDomainRequest [_ delete-domain-request]
  (let [{:keys [domain-name]} delete-domain-request]
    (.build
     (doto (DeleteDomainRequest/builder)
       (.domainName (name domain-name))))))

(defmethod ->sdk ListDomainsRequest [_ list-domains-request]
  (let [{:keys [max-number-of-domains
                next-token]} list-domains-request]
    (.build
     (doto (ListDomainsRequest/builder)
       (cond-> max-number-of-domains (.maxNumberOfDomains (int max-number-of-domains))
               next-token (.nextToken next-token))))))

(defmethod ->sdk DeleteAttributesRequest [_ delete-attributes-request]
  (let [{:keys [domain-name
                item-name
                attributes
                expected]} delete-attributes-request]
    (.build
     (doto (DeleteAttributesRequest/builder)
       (.domainName (name domain-name))
       (.itemName item-name)
       (cond->
           expected   (.expected ^UpdateCondition (->sdk UpdateCondition expected))
           attributes (.attributes
                       #^"[Lsoftware.amazon.awssdk.services.simpledb.model.Attribute;"
                       (into-array Attribute (mapv (partial ->sdk Attribute) attributes))))))))

(defmethod ->sdk DomainMetadataRequest [_ domain-metadata-request]
  (let [{:keys [domain-name]} domain-metadata-request]
    (.build
     (doto (DomainMetadataRequest/builder)
       (.domainName (name domain-name))))))

(defmethod ->sdk GetAttributesRequest [_ get-attributes-request]
  (let [{:keys [attribute-names
                consistent-read
                domain-name
                item-name]
         :or {consistent-read true}} get-attributes-request]
    (.build
     (doto (GetAttributesRequest/builder)
       (.attributeNames
        #^"[Ljava.lang.String;"
        (into-array String attribute-names))
       (.consistentRead consistent-read)
       (.domainName (name domain-name))
       (.itemName item-name)))))

(defmethod ->sdk PutAttributesRequest [_ put-attributes-request]
  (let [{:keys [domain-name
                item-name
                attributes
                expected]} put-attributes-request]
    (.build
     (doto (PutAttributesRequest/builder)
       (.domainName (name domain-name))
       (.itemName item-name)
       (.attributes
        #^"[Lsoftware.amazon.awssdk.services.simpledb.model.ReplaceableAttribute;"
        (into-array ReplaceableAttribute (mapv (partial ->sdk ReplaceableAttribute) attributes)))
       (cond->
           expected (.expected ^UpdateCondition (->sdk UpdateCondition expected)))))))

(defmethod ->sdk ReplaceableAttribute [_ replaceable-attribute]
  (let [{:keys [name
                replace
                value]} replaceable-attribute]
    (.build
     (doto (ReplaceableAttribute/builder)
       (.name (clojure.core/name name))
       (.replace replace)
       (.value value)))))

(defmethod ->sdk SelectRequest [_ select-request]
  (let [{:keys [select-expression
                consistent-read
                next-token]
         :or {consistent-read true}} select-request]
    (.build
     (doto (SelectRequest/builder)
       (.selectExpression select-expression)
       (.consistentRead consistent-read)
       (.nextToken next-token)))))

(defmethod ->sdk UpdateCondition [_ update-condition]
  (let [{:keys [exists
                name
                value]} update-condition]
    (.build
     (doto (UpdateCondition/builder)
       (.exists (boolean exists))
       (.name (clojure.core/name name))
       (cond->
           value (.value value))))))
