(ns aws.ssm.coercion
  (:require [aws.auth :as auth]
            [aws.client]
            [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.util :as u])
  (:import [software.amazon.awssdk.core.client.config ClientOverrideConfiguration]
           [software.amazon.awssdk.http SdkHttpClient]
           [software.amazon.awssdk.http.apache ApacheHttpClient]
           [software.amazon.awssdk.regions Region]
           [software.amazon.awssdk.services.ssm SsmClient]
           [software.amazon.awssdk.services.ssm.model
            GetParameterRequest
            GetParameterResponse
            GetParametersByPathRequest
            GetParametersByPathResponse
            GetParametersRequest
            GetParametersResponse
            PutParameterRequest
            PutParameterResponse
            Parameter
            ParameterStringFilter
            ParameterType]))
;; ----
;; client
;;
;; ----

(defmethod ->sdk SsmClient [_ ssm-client]
  (let [{:keys [http-client
                credentials-provider
                endpoint-override
                override-configuration
                region]
         :or {credentials-provider (auth/default-credentials-provider)}} ssm-client]
    (.build
     (doto (SsmClient/builder)
       (.credentialsProvider credentials-provider)
       (cond-> http-client (.httpClient ^SdkHttpClient (->sdk ApacheHttpClient http-client)))
       (cond-> region (.region (->sdk Region region)))
       (cond-> endpoint-override (.endpointOverride endpoint-override))
       (cond-> override-configuration (.overrideConfiguration ^ClientOverrideConfiguration (->sdk ClientOverrideConfiguration override-configuration)))))))

;; ----
;; to clojure
;;
;; ----

(extend-type GetParameterResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:parameter (->clj (.parameter response))})))

(extend-type GetParametersByPathResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:parameters (mapv (fn [p] (->clj p)) (.parameters response))
      :next-token (.nextToken response)})))

(extend-type GetParametersResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:parameters (mapv (fn [p] (->clj p)) (.parameters response))
      :invalid-parameters (into [] (.invalidParameters response))})))

(extend-type PutParameterResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:version (.version response)})))

(extend-type Parameter
  aws.coerce.to-clj/ToClojure
  (to-clj [parameter]
    (u/only-valid-values
     {:name (.name parameter)
      :type (.typeAsString parameter)
      :value (.value parameter)
      :versoin (.version parameter)})))

;; ----
;; to sdk
;;
;; ---

(defmethod ->sdk GetParameterRequest [_ get-parameter-request]
  (let [{:keys [name
                with-decryption]
         :or {with-decryption true}} get-parameter-request]
    (.build
     (doto (GetParameterRequest/builder)
       (.name (clojure.core/name name))
       (.withDecryption with-decryption)))))

(defmethod ->sdk GetParametersByPathRequest [_ get-parameters-by-path-request]
  (let [{:keys [max-results
                next-token
                parameter-filters
                path
                recursive
                with-decryption]
         :or {max-results 10
              with-decryption true}} get-parameters-by-path-request]
    (.build
     (doto (GetParametersByPathRequest/builder)
       (.maxResults (int max-results))
       (.path path)
       (.withDecryption with-decryption)
       (cond-> next-token (.nextToken next-token)
               parameter-filters (.parameterFilters
                                  (into-array
                                   ParameterStringFilter
                                   (mapv
                                    (fn [sf]
                                      (->sdk ParameterStringFilter sf))
                                    parameter-filters)))
               (not (nil? recursive)) (.recursive recursive))))))

(defmethod ->sdk GetParametersRequest [_ get-parameters-request]
  (let [{:keys [names
                with-decryption]
         :or {with-decryption true}} get-parameters-request]
    (.build
     (doto (GetParametersRequest/builder)
       (.names (into-array String (mapv name names)))
       (.withDecryption with-decryption)))))

(defmethod ->sdk ParameterStringFilter [_ parameter-string-filter]
  (let [{:keys [key
                option
                values]
         :or {option "Equals"}} parameter-string-filter]
    (.build
     (doto (ParameterStringFilter/builder)
       (.key key)
       (.option option)
       (.values (into-array String values))))))

(defmethod ->sdk PutParameterRequest [_ put-parameter-request]
  (let [{:keys [allowed-pattern
                description
                key-id
                name
                overwrite
                type
                value]
         :or {overwrite true}} put-parameter-request]
    (.build
     (doto (PutParameterRequest/builder)
       (.name (clojure.core/name name))
       (.overwrite overwrite)
       (.type type)
       (.value (str value))
       (cond-> allowed-pattern (.allowedPattern allowed-pattern)
               description (.description description)
               key-id (.keyId key-id))))))
