(ns aws.sfn
  (:require [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.sfn.client :as client]
            [aws.sfn.coercion])
  (:import [software.amazon.awssdk.services.sfn SfnClient]
           [software.amazon.awssdk.services.sfn.model
            CreateActivityRequest
            CreateStateMachineRequest
            DeleteActivityRequest
            DeleteStateMachineRequest
            DescribeActivityRequest
            DescribeStateMachineRequest
            GetActivityTaskRequest
            GetExecutionHistoryRequest
            ListActivitiesRequest
            ListExecutionsRequest
            ListStateMachinesRequest
            SendTaskFailureRequest
            SendTaskHeartbeatRequest
            SendTaskSuccessRequest
            StartExecutionRequest
            StopExecutionRequest
            UpdateStateMachineRequest]))

(defn create-activity [^SfnClient client create-activity-request]
  (->clj (.createActivity client ^CreateActivityRequest (->sdk CreateActivityRequest create-activity-request))))

(defn create-state-machine [^SfnClient clinet create-state-machine-request]
  (->clj (.createStateMachine clinet ^CreateStateMachineRequest (->sdk CreateStateMachineRequest create-state-machine-request))))

(defn delete-activity [^SfnClient client delete-activity-request]
  (->clj (.deleteActivity client ^DeleteActivityRequest (->sdk DeleteActivityRequest delete-activity-request))))

(defn delete-state-machine [^SfnClient client delete-state-machine-request]
  (->clj (.deleteStateMachine client ^DeleteStateMachineRequest (->sdk DeleteStateMachineRequest delete-state-machine-request))))

(defn describe-activity [^SfnClient client describe-activity-request]
  (->clj (.describeActivity client ^DescribeActivityRequest (->sdk DescribeActivityRequest describe-activity-request))))

(defn describe-state-machine [^SfnClient client describe-state-machine-request]
  (->clj (.describeStateMachine client ^DescribeStateMachineRequest (->sdk DescribeStateMachineRequest describe-state-machine-request))))

(defn get-activity-task [^SfnClient client get-activity-task-request]
  (->clj (.getActivityTask client ^GetActivityTaskRequest (->sdk GetActivityTaskRequest get-activity-task-request))))

(defn get-execution-history [^SfnClient client request]
  (let [list-fn (fn [req]
                  (->clj (.getExecutionHistory client ^GetExecutionHistoryRequest (->sdk GetExecutionHistoryRequest req))))

        next-batch-fn (fn [{:keys [next-token] :as response}]
                        (when next-token
                          (list-fn (assoc request :next-token next-token))))]

    (->>  (list-fn request)
          (iterate next-batch-fn)
          (take-while (complement nil?))
          (mapcat :events))))

(defn list-activities
  ([client] (list-activities client {}))
  ([^SfnClient client request]
   (let [list-fn (fn [req]
                   (->clj (.listActivities client ^ListActivitiesRequest (->sdk ListActivitiesRequest req))))

         next-batch-fn (fn [{:keys [next-token] :as response}]
                         (when next-token
                           (list-fn (assoc request :next-token next-token))))]

     (->>  (list-fn request)
           (iterate next-batch-fn)
           (take-while (complement nil?))
           (mapcat :activities)))))

(defn list-executions
  [^SfnClient client request]
  (let [list-fn (fn [req]
                  (->clj (.listExecutions client ^ListExecutionsRequest (->sdk ListExecutionsRequest req))))

        next-batch-fn (fn [{:keys [next-token] :as response}]
                        (when next-token
                          (list-fn (assoc request :next-token next-token))))]

    (->>  (list-fn request)
          (iterate next-batch-fn)
          (take-while (complement nil?))
          (mapcat :executions))))

(defn list-state-machines
  ([client] (list-state-machines client {}))
  ([^SfnClient client request]
   (let [list-fn (fn [req]
                   (->clj (.listStateMachines client ^ListStateMachinesRequest (->sdk ListStateMachinesRequest req))))

         next-batch-fn (fn [{:keys [next-token] :as response}]
                         (when next-token
                           (list-fn (assoc request :next-token next-token))))]

     (->>  (list-fn request)
           (iterate next-batch-fn)
           (take-while (complement nil?))
           (mapcat :state-machines)))))

(defn send-task-failure [^SfnClient client send-task-failure-request]
  (->clj (.sendTaskFailure client ^SendTaskFailureRequest (->sdk SendTaskFailureRequest send-task-failure-request))))

(defn send-task-heartbeat [^SfnClient client send-task-heartbeat-request]
  (->clj (.sendTaskHeartbeat client ^SendTaskHeartbeatRequest (->sdk SendTaskHeartbeatRequest send-task-heartbeat-request))))

(defn send-task-success [^SfnClient client send-task-success-request]
  (->clj (.sendTaskSuccess client ^SendTaskSuccessRequest (->sdk SendTaskSuccessRequest send-task-success-request))))

(defn start-execution [^SfnClient client start-execution-request]
  (->clj (.startExecution client ^StartExecutionRequest (->sdk StartExecutionRequest start-execution-request))))

(defn stop-execution [^SfnClient client stop-execution-request]
  (->clj (.stopExecution client ^StopExecutionRequest (->sdk StopExecutionRequest stop-execution-request))))

(defn update-state-machine [^SfnClient client update-state-machine-request]
  (->clj (.updateStateMachine client ^UpdateStateMachineRequest (->sdk UpdateStateMachineRequest update-state-machine-request))))
