(ns aws.sfn.coercion
  (:require [aws.coerce.to-clj :refer [->clj]]
            [aws.coerce.to-sdk :refer [->sdk]]
            [aws.util :as u])
  (:import [software.amazon.awssdk.services.sfn.model
            ActivityFailedEventDetails
            ActivityListItem
            ActivityScheduledEventDetails
            ActivityScheduleFailedEventDetails
            ActivityStartedEventDetails
            ActivitySucceededEventDetails
            ActivityTimedOutEventDetails
            CreateActivityRequest
            CreateActivityResponse
            CreateStateMachineRequest
            CreateStateMachineResponse
            DeleteActivityRequest
            DeleteActivityResponse
            DeleteStateMachineRequest
            DeleteStateMachineResponse
            DescribeActivityRequest
            DescribeActivityResponse
            DescribeExecutionRequest
            DescribeExecutionResponse
            DescribeStateMachineRequest
            DescribeStateMachineResponse
            ExecutionAbortedEventDetails
            ExecutionFailedEventDetails
            ExecutionListItem
            ExecutionStartedEventDetails
            ExecutionSucceededEventDetails
            ExecutionTimedOutEventDetails
            GetActivityTaskRequest
            GetActivityTaskResponse
            GetExecutionHistoryRequest
            GetExecutionHistoryResponse
            HistoryEvent
            HistoryEventType
            LambdaFunctionFailedEventDetails
            LambdaFunctionScheduleFailedEventDetails
            LambdaFunctionScheduledEventDetails
            LambdaFunctionStartFailedEventDetails
            LambdaFunctionSucceededEventDetails
            LambdaFunctionTimedOutEventDetails
            ListActivitiesRequest
            ListActivitiesResponse
            ListExecutionsRequest
            ListExecutionsResponse
            ListStateMachinesRequest
            ListStateMachinesResponse
            SendTaskFailureRequest
            SendTaskFailureResponse
            SendTaskHeartbeatRequest
            SendTaskHeartbeatResponse
            SendTaskSuccessRequest
            SendTaskSuccessResponse
            StartExecutionRequest
            StartExecutionResponse
            StateEnteredEventDetails
            StateExitedEventDetails
            StateMachineListItem
            StopExecutionRequest
            StopExecutionResponse
            UpdateStateMachineRequest
            UpdateStateMachineResponse]))

;; ----
;; HistoryEvent details to clojure
;;
;; ----

(defn state-entered [^HistoryEvent event]
  (->clj ^StateEnteredEventDetails (.stateEnteredEventDetails event)))

(defn state-exited [^HistoryEvent event]
  (->clj ^StateExitedEventDetails (.stateExitedEventDetails event)))

(defn event-details-dispatch-fn [^HistoryEvent history-event]
  (keyword (str (.type history-event))))

(defmulti event-details #'event-details-dispatch-fn)

(defmethod event-details :ActivityFailed [^HistoryEvent event]
  (->clj ^ActivityFailedEventDetails (.activityFailedEventDetails event)))

(defmethod event-details :ActivityScheduled [^HistoryEvent event]
  (->clj ^ActivityScheduledEventDetails (.activityScheduledEventDetails event)))

(defmethod event-details :ActivityScheduleFailed [^HistoryEvent event]
  (->clj ^ActivityScheduleFailedEventDetails (.activityScheduleFailedEventDetails event)))

(defmethod event-details :ActivityStarted [^HistoryEvent event]
  (->clj ^ActivityStartedEventDetails (.activityStartedEventDetails event)))

(defmethod event-details :ActivitySucceeded [^HistoryEvent event]
  (->clj ^ActivitySucceededEventDetails (.activitySucceededEventDetails event)))

(defmethod event-details :ActivityTimedOut [^HistoryEvent event]
  (->clj ^ActivityTimedOutEventDetails (.activityTimedOutEventDetails event)))

(defmethod event-details :ChoiceStateEntered [^HistoryEvent event]
  (state-entered event))

(defmethod event-details :ChoiceStateExited [^HistoryEvent event]
  (state-exited event))

(defmethod event-details :ExecutionAborted [^HistoryEvent event]
  (->clj ^ExecutionAbortedEventDetails (.executionAbortedEventDetails event)))

(defmethod event-details :ExecutionFailed [^HistoryEvent event]
  (->clj ^ExecutionFailedEventDetails (.executionFailedEventDetails event)))

(defmethod event-details :ExecutionStarted [^HistoryEvent event]
  (->clj ^ExecutionStartedEventDetails (.executionStartedEventDetails event)))

(defmethod event-details :ExecutionSucceeded [^HistoryEvent event]
  (->clj ^ExecutionSucceededEventDetails (.executionSucceededEventDetails event)))

(defmethod event-details :ExecutionTimedOut [^HistoryEvent event]
  (->clj ^ExecutionTimedOutEventDetails (.executionTimedOutEventDetails event)))

(defmethod event-details :FailStateEntered [^HistoryEvent event]
  (state-entered event))

(defmethod event-details :LambdaFunctionFailed [^HistoryEvent event]
  (->clj ^LambdaFunctionFailedEventDetails (.lambdaFunctionFailedEventDetails event)))

(defmethod event-details :LambdaFunctionScheduleFailed [^HistoryEvent event]
  (->clj ^LambdaFunctionScheduleFailedEventDetails (.lambdaFunctionScheduleFailedEventDetails event)))

(defmethod event-details :LambdaFunctionScheduled [^HistoryEvent event]
  (->clj ^LambdaFunctionScheduledEventDetails (.lambdaFunctionScheduledEventDetails event)))

(defmethod event-details :LambdaFunctionStartFailed [^HistoryEvent event]
  (->clj ^LambdaFunctionStartFailedEventDetails (.lambdaFunctionStartFailedEventDetails event)))

(defmethod event-details :LambdaFunctionStarted [^HistoryEvent event])

(defmethod event-details :LambdaFunctionSucceeded [^HistoryEvent event]
  (->clj ^LambdaFunctionSucceededEventDetails (.lambdaFunctionSucceededEventDetails event)))

(defmethod event-details :LambdaFunctionTimedOut [^HistoryEvent event]
  (->clj ^LambdaFunctionTimedOutEventDetails (.lambdaFunctionTimedOutEventDetails event)))

(defmethod event-details :SucceedStateEntered [^HistoryEvent event]
  (state-entered event))

(defmethod event-details :SucceedStateExited [^HistoryEvent event]
  (state-exited event))

(defmethod event-details :TaskStateAborted [^HistoryEvent event])

(defmethod event-details :TaskStateEntered [^HistoryEvent event]
  (state-entered event))

(defmethod event-details :TaskStateExited [^HistoryEvent event]
  (state-exited event))

(defmethod event-details :PassStateEntered [^HistoryEvent event]
  (state-entered event))

(defmethod event-details :PassStateExited [^HistoryEvent event]
  (state-exited event))

(defmethod event-details :ParallelStateAborted [^HistoryEvent event])

(defmethod event-details :ParallelStateEntered [^HistoryEvent event]
  (state-entered event))

(defmethod event-details :ParallelStateExited [^HistoryEvent event]
  (state-exited event))

(defmethod event-details :ParallelStateFalied [^HistoryEvent event])

(defmethod event-details :ParallelStateStarted [^HistoryEvent event])

(defmethod event-details :ParallelStateSucceeded [^HistoryEvent event])

(defmethod event-details :WaitStateAborted [^HistoryEvent event])

(defmethod event-details :WaitStateEntered [^HistoryEvent event]
  (state-entered event))

(defmethod event-details :WaitStateExited [^HistoryEvent event]
  (state-exited event))

(defmethod event-details :default [^HistoryEvent event]
  {:state "unknown to SDK"})

;; ----
;; to clojure
;;
;; ----

(extend-type ActivityFailedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type ActivityListItem
  aws.coerce.to-clj/ToClojure
  (to-clj [item]
    (u/only-valid-values
     {:activity-arn (.activityArn item)
      :creation-date (.creationDate item)
      :name (.name item)})))

(extend-type ActivityScheduledEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:heartbeat-in-seconds (.heartbeatInSeconds details)
      :input (.input details)
      :resource (.resource details)
      :timeout-in-seconds (.timeoutInSeconds details)})))

(extend-type ActivityScheduleFailedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type ActivityStartedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:worker-name (.workerName details)})))

(extend-type ActivitySucceededEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:output (.output details)})))

(extend-type ActivityTimedOutEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type CreateActivityResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:activity-arn (.activityArn response)
      :creation-date (.creationDate response)})))

(extend-type CreateStateMachineResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:state-machine-arn (.stateMachineArn response)
      :creation-date (.creationDate response)})))

(extend-type DeleteActivityResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    {}))

(extend-type DeleteStateMachineResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    {}))

(extend-type DescribeActivityResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:activity-arn (.activityArn response)
      :creation-date (.creationDate response)
      :name (.name response)})))

(extend-type DescribeExecutionResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:execution-arn (.executionArn response)
      :input (.input response)
      :name (.name response)
      :output (.output response)
      :start-date (.startDate response)
      :state-machine-arn (.stateMachineArn response)
      :status (.status response)
      :stop-date (.stopDate response)})))

(extend-type DescribeStateMachineResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:creation-date (.creationDate response)
      :definition (.definition response)
      :name (.name response)
      :role-arn (.roleArn response)
      :state-machine-arn (.stateMachineArn response)
      :status (.statusAsString response)})))

(extend-type ExecutionAbortedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type ExecutionFailedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type ExecutionListItem
  aws.coerce.to-clj/ToClojure
  (to-clj [item]
    (u/only-valid-values
     {:execution-arn (.executionArn item)
      :name (.name item)
      :start-date (.startDate item)
      :state-machine-arn (.stateMachineArn item)
      :status (.statusAsString item)
      :stop-date (.stopDate item)})))

(extend-type ExecutionStartedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:input (.input details)
      :role-arn (.roleArn details)})))

(extend-type ExecutionSucceededEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:output (.output details)})))

(extend-type ExecutionTimedOutEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type GetActivityTaskResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:task-token (.taskToken response)
      :input (.input response)})))

(extend-type GetExecutionHistoryResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:events (mapv (fn [e] (->clj e)) (.events response))
      :next-token (.nextToken response)})))

(extend-type HistoryEvent
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:event-details (event-details response)
      :id (.id response)
      :previous-event-id (.previousEventId response)
      :timestamp (.timestamp response)
      :type (.typeAsString response)})))

(extend-type LambdaFunctionFailedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type LambdaFunctionScheduleFailedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type LambdaFunctionScheduledEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:input (.input details)
      :resource (.resource details)
      :timeout-in-seconds (.timeoutInSeconds details)})))

(extend-type LambdaFunctionStartFailedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type LambdaFunctionSucceededEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:output (.output details)})))

(extend-type LambdaFunctionTimedOutEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:cause (.cause details)
      :error (.error details)})))

(extend-type ListActivitiesResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:activities (mapv (fn [activity-list-item](->clj activity-list-item)) (.activities response))
      :next-token (.nextToken response)})))

(extend-type ListExecutionsResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:executions (mapv (fn [execution-list-item](->clj execution-list-item)) (.executions response))
      :next-token (.nextToken response)})))

(extend-type ListStateMachinesResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:state-machines (mapv (fn [state-machine-list-item](->clj state-machine-list-item)) (.stateMachines response))
      :next-token (.nextToken response)})))

(extend-type SendTaskFailureResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    {}))

(extend-type SendTaskHeartbeatResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    {}))

(extend-type SendTaskSuccessResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    {}))

(extend-type StartExecutionResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:execution-arn (.executionArn response)
      :start-date (.startDate response)})))

(extend-type StateEnteredEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:input (.input details)
      :name (.name  details)})))

(extend-type StateExitedEventDetails
  aws.coerce.to-clj/ToClojure
  (to-clj [details]
    (u/only-valid-values
     {:output (.output details)
      :name (.name  details)})))

(extend-type StateMachineListItem
  aws.coerce.to-clj/ToClojure
  (to-clj [item]
    (u/only-valid-values
     {:creation-date (.creationDate item)
      :name (.name item)
      :state-machine-arn (.stateMachineArn item)})))

(extend-type StopExecutionResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:stop-date (.stopDate response)})))

(extend-type UpdateStateMachineResponse
  aws.coerce.to-clj/ToClojure
  (to-clj [response]
    (u/only-valid-values
     {:update-date (.updateDate response)})))

;; ----
;; to sdk
;;
;; ---

(defmethod ->sdk CreateActivityRequest [_ create-activity-request]
  (let [{:keys [name]} create-activity-request]
    (.build
     (doto (CreateActivityRequest/builder)
       (.name (clojure.core/name name))))))

(defmethod ->sdk CreateStateMachineRequest [_ create-state-machine-request]
  (let [{:keys [definition
                name
                role-arn]} create-state-machine-request]
    (.build
     (doto (CreateStateMachineRequest/builder)
       (.definition ^String definition)
       (.name (clojure.core/name name))
       (.roleArn role-arn)))))

(defmethod ->sdk DeleteActivityRequest [_ delete-activity-request]
  (let [{:keys [activity-arn]} delete-activity-request]
    (.build
     (doto (DeleteActivityRequest/builder)
       (.activityArn activity-arn)))))

(defmethod ->sdk DeleteStateMachineRequest [_ delete-state-machine-request]
  (let [{:keys [state-machine-arn]} delete-state-machine-request]
    (.build
     (doto (DeleteStateMachineRequest/builder)
       (.stateMachineArn state-machine-arn)))))

(defmethod ->sdk DescribeActivityRequest [_ describe-activity-request]
  (let [{:keys [activity-arn]} describe-activity-request]
    (.build
     (doto (DescribeActivityRequest/builder)
       (.activityArn activity-arn)))))

(defmethod ->sdk DescribeExecutionRequest [_ describe-execution-request]
  (let [{:keys [execution-arn]} describe-execution-request]
    (.build
     (doto (DescribeExecutionRequest/builder)
       (.executionArn execution-arn)))))

(defmethod ->sdk DescribeStateMachineRequest [_ describe-state-machine-request]
  (let [{:keys [state-machine-arn]} describe-state-machine-request]
    (.build
     (doto (DescribeStateMachineRequest/builder)
       (.stateMachineArn state-machine-arn)))))

(defmethod ->sdk GetActivityTaskRequest [_ get-activity-task-request]
  (let [{:keys [activity-arn
                worker-name]} get-activity-task-request]
    (.build
     (doto (GetActivityTaskRequest/builder)
       (.activityArn activity-arn)
       (.workerName worker-name)))))

(defmethod ->sdk GetExecutionHistoryRequest [_ get-execution-history-request]
  (let [{:keys [execution-arn
                max-results
                next-token
                reverse-order]
         :or {max-results 100
              reverse-order false}} get-execution-history-request]
    (.build
     (doto (GetExecutionHistoryRequest/builder)
       (.executionArn execution-arn)
       (.maxResults (int max-results))
       (.reverseOrder reverse-order)
       (cond-> next-token (.nextToken next-token))))))

(defmethod ->sdk ListActivitiesRequest [_ list-activities-request]
  (let [{:keys [max-results
                next-token]
         :or {max-results 100}} list-activities-request]
    (.build
     (doto (ListActivitiesRequest/builder)
       (.maxResults (int max-results))
       (cond-> next-token (.nextToken  next-token))))))

(defmethod ->sdk ListExecutionsRequest [_ list-executions-request]
  (let [{:keys [max-results
                next-token
                state-machine-arn
                status-filter]
         :or {max-results 100}} list-executions-request]
    (.build
     (doto (ListExecutionsRequest/builder)
       (.stateMachineArn state-machine-arn)
       (.maxResults (int max-results))
       (cond-> next-token (.nextToken  next-token)
               status-filter (.statusFilter ^String (str status-filter)))))))

(defmethod ->sdk ListStateMachinesRequest [_ list-state-machine-request]
  (let [{:keys [max-results
                next-token]
         :or {max-results 100}} list-state-machine-request]
    (.build
     (doto (ListStateMachinesRequest/builder)
       (.maxResults (int max-results))
       (cond-> next-token (.nextToken  next-token))))))

(defmethod ->sdk SendTaskFailureRequest [_ send-task-failure-request]
  (let [{:keys [cause
                error
                task-token]} send-task-failure-request]
    (.build
     (doto (SendTaskFailureRequest/builder)
       (.cause cause)
       (.error error)
       (.taskToken task-token)))))

(defmethod ->sdk SendTaskHeartbeatRequest [_ send-task-heartbeat-request]
  (let [{:keys [task-token]} send-task-heartbeat-request]
    (.build
     (doto (SendTaskHeartbeatRequest/builder)
       (.taskToken task-token)))))

(defmethod ->sdk SendTaskSuccessRequest [_ send-task-success-request]
  (let [{:keys [output
                task-token]} send-task-success-request]
    (.build
     (doto (SendTaskSuccessRequest/builder)
       (.output output)
       (.taskToken task-token)))))

(defmethod ->sdk StartExecutionRequest [_ start-execution-request]
  (let [{:keys [input
                name
                state-machine-arn]} start-execution-request]
    (.build
     (doto (StartExecutionRequest/builder)
       (.input input)
       (.name (clojure.core/name name))
       (.stateMachineArn state-machine-arn)))))

(defmethod ->sdk StopExecutionRequest [_ stop-execution-request]
  (let [{:keys [cause
                error
                execution-arn]} stop-execution-request]
    (.build
     (doto (StopExecutionRequest/builder)
       (.cause cause)
       (.error error)
       (.executionArn execution-arn)))))

(defmethod ->sdk UpdateStateMachineRequest [_ update-state-machine-request]
  (let [{:keys [definition
                role-arn
                state-machine-arn]} update-state-machine-request]
    (.build
     (doto (UpdateStateMachineRequest/builder)
       (.definition definition)
       (.roleArn role-arn)
       (.stateMachineArn state-machine-arn)))))
