(ns beam-es.core
  (:require [cheshire.core :as json]
            [taoensso.timbre :as log]
            [clj-http.client :as client]))

(def es-chunk-size 5000)

(defn- es-put-bulk-uri
  [uri]
  (str "http://" uri "/_bulk"))

(defn- es-bulk-entry
  "Creates a clojure map for a bulk entry. This function creates Elasticsearch
  index row and doc row based on Elasticsearch format here:
  https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-bulk.html.

  If `id` is not provided, Elasticsearch will create automatically."
  ([index id doc]
   {:action {:index
             {:_index index
              :_type  "_doc"
              :_id    id}}
    :doc    doc})
  ([index doc]
   {:action {:index
             {:_index index
              :_type  "_doc"}}
    :doc    doc}))

(defn- es-bulk-entries
  "Builds list of objects that elasticsearch expects for a bulk insert.
  Objects are subsequently stringified by `es-bulk-body-string`."
  ([index id data]
   (map #(es-bulk-entry index id %) data))
  ([index data]
   (map #(es-bulk-entry index %) data)))

(defn- es-bulk-body-string
  "Returns a string suitable for using as :body element in elasticsearch bulk
  index request"
  [bulk-insert-entries]
  (reduce (fn [result-str entry]
            (str result-str (json/generate-string (:action entry)) "\n"
                 (json/generate-string (:doc entry)) "\n")) "" bulk-insert-entries))

(defn- chunked-es-bulk-body-strings
  "Chunk up the bulk insert list to ensure that bulk insert is not being
  stretched. Bulk inserts are chunked into `es-chunk-size` (5,000) chunks."
  [index data]
  (->> data
       (es-bulk-entries index)
       (partition-all es-chunk-size,,,)
       (map es-bulk-body-string,,,)))

(defn es-post-bulk!
  [es-uri index data]
  (log/info "Writing " (count data) " docs to " es-uri " elasticsearch instance")
  (doseq [body-string (chunked-es-bulk-body-strings index data)]
    (client/post (es-put-bulk-uri es-uri)
                 {:content-type :json
                  :body         body-string})))

(defn criteria->es-match-phrase-query
  "Converts criteria map to Elasticsearch match_phrase query format"
  [criteria]
  (let [match-phrase-query
        (map (fn [[key val]]
               (hash-map :match_phrase {key val})) criteria)]
    {:query {:bool {:must match-phrase-query}}}))

(defn es-delete-by-query!
  "Deletes docs from elasticsearch based on `criteria`.
  `criteria` is a map with multiple `field = match-value` search criteria."
  [es-uri index criteria]
  (log/info "Deleting "  " docs from " es-uri " elasticsearch instance with criteria " criteria)
  (let [body (criteria->es-match-phrase-query criteria)
        body-str (json/generate-string body)]
    (client/post (str "http://" es-uri "/" index "/_delete_by_query")
                 {:content-type :json
                  :body body-str})))

;(es-delete-by-query! "localhost:1337" "device-event" {:device-id "NMI-43103616674"
;                                                      :time-stamp "2017-05-22T17:15Z"})