(ns beam-es.core
  (:require [cheshire.core :as json]
            [taoensso.timbre :as log]
            [clj-http.client :as client]))

(def chunk-size 1000)

(defn- put-bulk-uri
  [uri]
  (str "http://" uri "/_bulk"))

(defn- put-uri
  [es-uri index id]
  (str "http://" es-uri "/" index "/doc/" id))

(defn- bulk-entry
  "Creates a clojure map for a bulk entry. This function creates Elasticsearch
  index row and doc row based on Elasticsearch format here:
  https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-bulk.html.

  If `id` is not provided, Elasticsearch will create automatically."
  ([index id doc]
   {:action {:index
             {:_index index
              :_type  "doc"
              :_id    id}}
    :doc    doc})
  ([index doc]
   {:action {:index
             {:_index index
              :_type  "doc"}}
    :doc    doc}))

(defn- bulk-entries
  "Builds list of objects that elasticsearch expects for a bulk insert.
  Objects are subsequently stringified by `es-bulk-body-string`."
  ([index id data]
   (map #(bulk-entry index id %) data))
  ([index data]
   (map #(bulk-entry index %) data)))

(defn- bulk-body-string
  "Returns a string suitable for using as :body element in elasticsearch bulk
  index request"
  [bulk-insert-entries]
  (reduce (fn [result-str entry]
            (str result-str (json/generate-string (:action entry)) "\n"
                 (json/generate-string (:doc entry)) "\n")) "" bulk-insert-entries))

(defn- chunked-bulk-body-strings
  "Chunk up the bulk insert list to ensure that bulk insert is not being
  stretched. Bulk inserts are chunked into `es-chunk-size` (5,000) chunks."
  [index data]
  (->> data
       (bulk-entries index)
       (partition-all chunk-size,,,)
       (map bulk-body-string,,,)))

;; TODO check es result for successful inserts and throw error on failure
(defn post-bulk!
  [es-uri index data]
  (log/info "Writing" (count data) "docs to" index "in elasticsearch")
  (let [bulk-uri (put-bulk-uri es-uri)
        bodies (chunked-bulk-body-strings index data)]
    (->> (map (fn [body-string]
                (future (client/post bulk-uri
                                     {:content-type :json
                                      :body         body-string}))) bodies)
         (map deref ,,,)
         count)))

(defn put!
  "Puts doc to elasticsearch"
  [es-uri index id doc]
  (let [uri (put-uri es-uri index id)]
    (client/put uri {:content-type :json
                     :body         (json/generate-string doc)})))

(defn- criteria->query
  "Converts criteria map to Elasticsearch match_phrase query format.
  If provided with `start` `end` and `timezone` params, returns a query
  object that includes `range` query of epoc element."
  ([criteria]
   (let [match-phrase-query
         (map (fn [[key val]]
                (hash-map :match_phrase {key val})) criteria)]
     {:query {:bool {:must match-phrase-query}}}))
  ([criteria start end timezone]
   (let [match-phrase-query
         (mapv (fn [[key val]]
                 (hash-map :match_phrase {key val})) criteria)
         range-query {:range {:epoch {:gte start :lte end :time_zone timezone}}}]
     {:query {:bool {:must (conj match-phrase-query range-query)}}})))

(defn- histogram-agg-query
  "Creates an aggs elasticsearch query object based on `field-list`"
  [interval timezone field-list]
  (let [fields (reduce (fn [m val] (into m {(keyword val) {:sum {:field (name val)}}})) {} field-list)]
    {:aggs {:month {:date_histogram {:field "epoch",
                                     :interval interval,
                                     :time_zone timezone},
                    :aggs fields}}}))

(defn- flat-bucket
  "Flattens histogram-bucket and returns a flat map with `field-list` items"
  [bucket field-list]
  (let [field-list (map keyword field-list)
        field-values (map #(get-in bucket [% :value]) field-list)
        clean-bucket (dissoc bucket field-list)]
    (merge clean-bucket (zipmap field-list field-values))))

(defn delete-by-query!
  "Deletes docs from elasticsearch based on `criteria`.
  `criteria` is a map with multiple `field = match-value` search criteria."
  [es-uri index criteria]
  (log/info "Deleting docs from elasticsearch index:" index "with criteria:" criteria)
  (let [body (criteria->query criteria)
        body-str (json/generate-string body)]
    (client/post (str "http://" es-uri "/" index "/_delete_by_query?conflicts=proceed")
                 {:content-type :json
                  :body         body-str})))

(defn parse-histogram-result
  "Extracts the required values from elasticsearch result and converts to
  a simple flat array."
  [es-result field-list]
  (let [histogram-buckets (get-in es-result [:aggregations :month :buckets])]
    (->> (map #(flat-bucket % field-list) histogram-buckets)
         (map #(assoc % :timestamp (:key_as_string %)) ,,,)
         (map #(dissoc % :key_as_string)) ,,,)))

(defn date-histogram!
  "Returns a date histogram based on params:
  `interval` hour | day | week | month | year,
  `start` a local date - for example `2017-05-04`
  `end` local date
  `timezone` for example `Australia/Sydney`
  `criteria` a key value map of field critera to search by.
  `field-list` a keyword/string list of field names to aggregate."
  [es-uri index interval start end timezone criteria field-list]
  (log/info "Requesting data histogram from elasticsearch. Index:" index "timespan:" interval "start:" start "end:" end "timezone:" timezone)
  (let [body-string (json/generate-string
                      (merge
                        (histogram-agg-query interval timezone field-list)
                        (criteria->query criteria start end timezone)))]
    (-> (client/get (str "http://" es-uri "/" index "/_search?size=0")
                    {:content-type :json
                     :body body-string})
        :body
        (json/parse-string ,,, true)
        (parse-histogram-result ,,, field-list))))

(defn- parse-query-result
  "Parses elasticsearch query result returning just the
  hit objects as a list of clojure maps."
  [query-result]
  (->> (get-in query-result [:hits :hits])
       (map :_source ,,,)))

(defn query!
  "Returns docs from `index` based on `criteria`.
  `criteria` in form of a clojure map.
  Max return size set to 1000."
  [es-uri index criteria]
  (log/info "Querying docs from elasticsearch index:" index "with criteria:" criteria)
  (let [size 1000
        criteria (criteria->query criteria)
        body (-> (assoc criteria :size size) json/generate-string)]
    (-> (client/post (str "http://" es-uri "/" index "/_search?")
                     {:content-type :json
                      :body         body})
        :body
        (json/parse-string ,,, true)
        parse-query-result)))

;(time (es-date-histogram! "localhost:1337" "timeseries-data" "week"
;                          "2017-04-01" "2018-05-02" "Australia/Perth"
;                          {:device-id "NMI-8001010350" :scenario-id 1}
;                          [:consumption-ascribed-Wh :solar-simulated-wh :solar-used-wh :solar-sold-wh :battery-charged-wh :battery-used-wh :grid-used-wh]))

;(es-delete-by-query! "localhost:1337" "device-event" {:device-id "NMI-43103616674"
;                                                      :time-stamp "2017-05-22T17:15Z"})