(ns chromex.ext.wallpaper
  "Use the chrome.wallpaper API to change the ChromeOS wallpaper.

     * available since Chrome 31
     * https://developer.chrome.com/extensions/wallpaper"

  (:refer-clojure :only [defmacro defn apply declare meta let])
  (:require [chromex.wrapgen :refer [gen-wrap-from-table]]
            [chromex.callgen :refer [gen-call-from-table gen-tap-all-call]]
            [chromex.config :refer [get-static-config gen-active-config]]))

(declare api-table)
(declare gen-call)

; -- functions --------------------------------------------------------------------------------------------------------------

(defmacro set-wallpaper
  "Sets wallpaper to the image at url or wallpaperData with the specified layout

     |details| - https://developer.chrome.com/extensions/wallpaper#property-setWallpaper-details.

   This function returns a core.async channel which eventually receives a result value and closes.
   Signature of the result value put on the channel is [thumbnail] where:

     |thumbnail| - The jpeg encoded wallpaper thumbnail. It is generated by resizing the wallpaper to 128x60.

   https://developer.chrome.com/extensions/wallpaper#method-setWallpaper."
  ([details] (gen-call :function ::set-wallpaper &form details)))

; -- convenience ------------------------------------------------------------------------------------------------------------

(defmacro tap-all-events
  "Taps all valid non-deprecated events in this namespace."
  [chan]
  (let [static-config (get-static-config)
        config (gen-active-config static-config)]
    (gen-tap-all-call static-config api-table (meta &form) config chan)))

; ---------------------------------------------------------------------------------------------------------------------------
; -- API TABLE --------------------------------------------------------------------------------------------------------------
; ---------------------------------------------------------------------------------------------------------------------------

(def api-table
  {:namespace "chrome.wallpaper",
   :since "31",
   :functions
   [{:id ::set-wallpaper,
     :name "setWallpaper",
     :since "33",
     :callback? true,
     :params
     [{:name "details", :type "object"}
      {:name "callback",
       :type :callback,
       :callback {:params [{:name "thumbnail", :optional? true, :type "binary"}]}}]}]})

; -- helpers ----------------------------------------------------------------------------------------------------------------

; code generation for native API wrapper
(defmacro gen-wrap [kind item-id config & args]
  (let [static-config (get-static-config)]
    (apply gen-wrap-from-table static-config api-table kind item-id config args)))

; code generation for API call-site
(defn gen-call [kind item src-info & args]
  (let [static-config (get-static-config)
        config (gen-active-config static-config)]
    (apply gen-call-from-table static-config api-table kind item src-info config args)))