(ns chromex.ext.platform-keys-internal
  "Internal API for to implement the platformKeys and enterprise.platformKeys
   APIs.

     * available since Chrome 42"

  (:refer-clojure :only [defmacro defn apply declare meta let partial])
  (:require [chromex.wrapgen :refer [gen-wrap-helper]]
            [chromex.callgen :refer [gen-call-helper gen-tap-all-events-call]]))

(declare api-table)
(declare gen-call)

; -- functions --------------------------------------------------------------------------------------------------------------

(defmacro select-client-certificates
  "See documentation in platformKeys.

     |details| - ?

   This function returns a core.async channel which eventually receives a result value and closes.
   Signature of the result value put on the channel is [certs] where:

     |certs| - ?

   In case of error the channel closes without receiving any result and relevant error object can be obtained via
   chromex.error/get-last-error."
  ([details] (gen-call :function ::select-client-certificates &form details)))

(defmacro sign
  "Internal version of platformKeys.subtleCrypto.sign and enterprise.platformKeys.Token.subtleCrypto.sign. |tokenId| The id of
   a Token returned by |getTokens|. |publicKey| The Subject Public Key Info of a key previously generated by     |generateKey

   in DER encoding. |hashAlgorithmName| The recognized algorithm name of the hash algorithm,     as specified by WebCrypto,
   that will be used to digest |data|     before signing. Currently supported are: SHA-{1,256,384,512}.     If instead the
   algorithm name 'none' is provided, no hashing will be     applied, the data is PKCS#1 v1.5 padded but not hashed.
   TODO(pneubeck): use an enum once supported:         http://www.crbug.com/385539 . |data| The data to sign. |callback

   Called back with the signature of |data|. TODO: Instead of ArrayBuffer should be (ArrayBuffer or ArrayBufferView), or at
   least (ArrayBuffer or Uint8Array).

     |token-id| - ?
     |public-key| - ?
     |hash-algorithm-name| - ?
     |data| - ?

   This function returns a core.async channel which eventually receives a result value and closes.
   Signature of the result value put on the channel is [signature] where:

     |signature| - ?

   In case of error the channel closes without receiving any result and relevant error object can be obtained via
   chromex.error/get-last-error."
  ([token-id public-key hash-algorithm-name data] (gen-call :function ::sign &form token-id public-key hash-algorithm-name data)))

(defmacro get-public-key
  "Checks whether certificate certifies a key that allows usage of the WebCrypto algorithm algorithmName. If so, calls back
   callback with the key info and a WebCrypto KeyAlgorithm dictionary describing the key's algorithm. The name property will
   equal algorithmName. Otherwise, calls back with an error.

     |certificate| - ?
     |algorithm-name| - ?

   This function returns a core.async channel which eventually receives a result value and closes.
   Signature of the result value put on the channel is [public-key algorithm] where:

     |public-key| - ?
     |algorithm| - ?

   In case of error the channel closes without receiving any result and relevant error object can be obtained via
   chromex.error/get-last-error."
  ([certificate algorithm-name] (gen-call :function ::get-public-key &form certificate algorithm-name)))

; -- convenience ------------------------------------------------------------------------------------------------------------

(defmacro tap-all-events
  "Taps all valid non-deprecated events in chromex.ext.platform-keys-internal namespace."
  [chan]
  (gen-tap-all-events-call api-table (meta &form) chan))

; ---------------------------------------------------------------------------------------------------------------------------
; -- API TABLE --------------------------------------------------------------------------------------------------------------
; ---------------------------------------------------------------------------------------------------------------------------

(def api-table
  {:namespace "chrome.platformKeysInternal",
   :since "42",
   :functions
   [{:id ::select-client-certificates,
     :name "selectClientCertificates",
     :callback? true,
     :params
     [{:name "details", :type "platformKeys.SelectDetails"}
      {:name "callback",
       :type :callback,
       :callback {:params [{:name "certs", :type "[array-of-platformKeys.Matchs]"}]}}]}
    {:id ::sign,
     :name "sign",
     :callback? true,
     :params
     [{:name "token-id", :type "string"}
      {:name "public-key", :type "ArrayBuffer"}
      {:name "hash-algorithm-name", :type "string"}
      {:name "data", :type "ArrayBuffer"}
      {:name "callback", :type :callback, :callback {:params [{:name "signature", :type "ArrayBuffer"}]}}]}
    {:id ::get-public-key,
     :name "getPublicKey",
     :callback? true,
     :params
     [{:name "certificate", :type "ArrayBuffer"}
      {:name "algorithm-name", :type "string"}
      {:name "callback",
       :type :callback,
       :callback {:params [{:name "public-key", :type "ArrayBuffer"} {:name "algorithm", :type "object"}]}}]}]})

; -- helpers ----------------------------------------------------------------------------------------------------------------

; code generation for native API wrapper
(defmacro gen-wrap [kind item-id config & args]
  (apply gen-wrap-helper api-table kind item-id config args))

; code generation for API call-site
(def gen-call (partial gen-call-helper api-table))