(ns cemerick.rummage.retry
  (:import [com.amazonaws
            AmazonClientException
            AmazonServiceException]
           [com.amazonaws.services.simpledb.model
            RequestTimeoutException]))

(defn with-retries* [retries-left sleep* retry? f]
  (let [r (try
            (f)
            (catch Throwable t
              t))]
    (cond
     ;; retry allowed and needed, retry
     (and (pos? retries-left) (retry? r))
     (let [sleep-ms (cond
                     (number? sleep*) sleep*
                     (fn? sleep*)     (sleep*)
                     :else            (throw (Exception. "Unsupported sleep*")))]
       (Thread/sleep sleep-ms)
       (recur (dec retries-left) sleep* retry? f))

     ;; not retrying, r is Throwable, re-throw r
     (instance? Throwable r)
     (throw r)

     ;; not retrying, r is a value, return r
     :else
     r)))

(defmacro with-retries
  "Executes forms, retrying if needed.

  If sleep* is a number, will wait sleep* millis between retries. If it's a
  function, it's repeatedly called to calculate wait between retries.

  retry? should be a predicate on the return value (or exception) from forms,
  returning true iff forms should be retried.

  Eg:

  (with-retries 3 100 (one-of? Exception)
     (println \"running\")
     (if (rand-nth [false true])
        (throw (Exception. \"Unlucky\"))
        :ok))

  (with-retries 3 (exponential-backoff 100) (one-of? Exception)
     (println \"running\")
     (if (rand-nth [false true])
        (throw (Exception. \"Unlucky\"))
        :ok))"
  [retries-left sleep* retry? & forms]
  `(with-retries* ~retries-left ~sleep* ~retry? #(do ~@forms)))

(defn exponential-backoff
  "Return a function that will return exponentially backed off times
   when called repeatedly. If multiplier is not specified, uses 2."
  [initial & [multiplier]]
  (let [m (or multiplier 2)
        x (atom initial)]
    #(let [r @x] ;; Safe to get then swap! since the atom is private.
       (swap! x * m)
       r)))

(defmacro with-sdb-retries
  "Macro to retry eligible SimpleDB errors."
  [& forms]
  `(with-retries 3 (exponential-backoff 100)
     (fn [r#]
       (or (instance? AmazonClientException r#)
           (instance? RequestTimeoutException r#)
           (and (instance? AmazonServiceException r#)
                (#{408 500 503} (.getStatusCode r#)))))
     ~@forms))
