(ns blueprints.clj.index
  (:refer-clojure :exclude [count get])
  (:require [blueprints.clj :as $])
  (:import (com.tinkerpop.blueprints Vertex Edge
                                     Index
                                     IndexableGraph KeyIndexableGraph
                                     Parameter
                                     Query)
           blueprints.clj.ElementWrapper
           clojure.lang.Keyword))

;; [Utils]
(defn ^:private ->parameters [hmap]
  (into-array (for [[k v] hmap]
                (Parameter. (if (keyword? k) (name k) k)
                            v))))

(defn ^:private ^Class ->class [kw]
  (case kw
    :vertex Vertex
    :edge   Edge))

;; [Interface]
;; Key indices
(defn indexed-keys=> "The set of indexed keys for :vertex|:edge."
  [type]
  (->> (.getIndexedKeys ^KeyIndexableGraph $/*db* (->class type))
       .iterator
       iterator-seq
       (map keyword)
       set))

(defn create-key-index! "" [^String key type & [params]]
  (.createKeyIndex ^KeyIndexableGraph $/*db* key (->class type) (->parameters params)))

(defn drop-key-index! "" [^String key type]
  (.dropKeyIndex ^KeyIndexableGraph $/*db* key (->class type)))

;; Regular indices
(defn indices=> "" []
  (-> (.getIndices ^IndexableGraph $/*db*)
      .iterator
      iterator-seq))

(defn index=> "" [^String idx-name type]
  (.getIndex ^IndexableGraph $/*db* idx-name (->class type)))

(defn create-index! "" [^String idx-name type & [params]]
  (.createIndex ^IndexableGraph $/*db* idx-name (->class type) (->parameters params)))

(defn drop-index! "" [^String idx-name]
  (.dropKeyIndex ^IndexableGraph $/*db* idx-name))

;; Index interface
(defn count "Count all the indexed elements with matching K/V."
  [^Index index ^Keyword key val]
  (.count index (name key) val))

(defn class=> "" [^Index index]
  (.getIndexClass index))

(defn name=> "" [^Index index]
  (.getIndexName index))

(defn get "Fetches element with K/V from index."
  [^Index index ^Keyword key val]
  (->> (.get index (name key) val)
       .iterator
       iterator-seq
       (map $/wrap)))

(defn put! "Adds element with K/V into index."
  [^Index index ^Keyword key val ^ElementWrapper elem]
  (.put index (name key) val (.obj elem)))

(defn remove! "Removes element with K/V from index."
  [^Index index ^Keyword key val ^ElementWrapper elem]
  (.remove index (name key) val (.obj elem)))

(defn query "Queries the index."
  [^Index index ^Keyword key query]
  (->> (.query index (name key) query)
       .iterator
       iterator-seq
       (map $/wrap)))
