(ns stripe.customer
  "Stripe's Customer API."
  (:require [schema.core :as s]
            [stripe.http :as h]
            [stripe.schema :as ss]
            [stripe.token :as t]))

;; ## Schema

(s/defschema CustomerID
  (s/named s/Str "The identifier of the customer to be retrieved."))

(s/defschema CardID
  (s/named s/Str "The identifier of the card."))

(s/defschema CustomerReq
  {(s/optional-key :card) t/Card
   (s/optional-key :description) s/Str
   (s/optional-key :email) s/Str
   (s/optional-key :metadata) ss/Metadata})

(s/defschema CardUpdate
  {(s/optional-key :address_city) s/Str
   (s/optional-key :address_country) s/Str
   (s/optional-key :address_line1) s/Str
   (s/optional-key :address_line2) s/Str
   (s/optional-key :address_state) s/Str
   (s/optional-key :address_zip) s/Str
   (s/optional-key :exp_month) t/ExpMonth
   (s/optional-key :exp_year) t/ExpYear
   (s/optional-key :metadata) ss/Metadata
   (s/optional-key :name) t/FullName})

;; ## Customer API Calls

(s/defn create-customer :- (ss/Async)
  "Creates a new customer using the Stripe API."
  [options :- CustomerReq]
  (h/post-req "customers" {:stripe-params options}))

(s/defn get-customer :- (ss/Async)
  "Retrieves the details of an existing customer. You need only supply
  the unique customer identifier that was returned upon customer
  creation."
  [id :- CustomerID]
  (h/get-req (str "customers/" id)))

(s/defn update-customer :- (ss/Async)
  "For our purposes, an update is the same as a creation. (The only
  API differences have to do with subscription management.)"
  [customer-id :- CustomerID
   options :- CustomerReq]
  (h/post-req (str "customers/" customer-id)
              {:stripe-params options}))

(s/defn delete-customer :- (ss/Async)
  "Deletes the supplied customer."
  [customer-id :- CustomerID]
  (h/delete-req (str "customers/" customer-id)))

(s/defn create-card :- (ss/Async)
  "Creates a new card associated with the user"
  [customer-id :- CustomerID
   card :- t/Card]
  (h/post-req (str "customers/" customer-id "/sources")
              {:stripe-params {:source card}}))

(s/defn delete-card :- (ss/Async)
  "Deletes a card from the customer's account."
  [customer-id :- CustomerID
   card-id :- CardID]
  (h/delete-req (str "customers/" customer-id "/sources/" card-id)))

(s/defn update-card :- (ss/Async)
  "Updates a customer's existing card."
  [customer-id :- CustomerID
   card-id :- CardID
   options :- CardUpdate]
  (h/post-req (str "customers/" customer-id "/sources/" card-id)
              {:stripe-params options}))
