(ns boot.main
  (:require
   [boot.cli             :as cli]
   [boot.pod             :as pod]
   [boot.core            :as core]
   [boot.file            :as file]
   [boot.util            :as util]
   [boot.gitignore       :as git]
   [boot.tmpregistry     :as tmp]
   [clojure.java.io      :as io]
   [clojure.string       :as string]))

(def boot-version (str (boot.App/getVersion) "-" (boot.App/getRelease)))

(defn read-cli [argv]
  (let [src (str "(" (string/join " " argv) "\n)")]
    (util/with-rethrow
      (read-string src)
      (format "Can't read command line as EDN: %s" src))))

(defn parse-cli [argv]
  (try (let [dfltsk `(((-> :default-task core/get-env resolve)))
             ->expr #(cond (seq? %) % (vector? %) (list* %) :else (list %))]
         (or (seq (map ->expr (or (seq (read-cli argv)) dfltsk))) dfltsk))
       (catch Throwable e (util/print-ex e))))

(def cli-opts
  [
   ["-b" "--boot-script"         "Print generated boot script for debugging."]
   ["-d" "--dependencies ID:VER" "Add dependency to project (eg. -d riddley:0.1.7)."
    :assoc-fn #(let [[p v] (string/split %3 #":" 2)]
                 (update-in %1 [%2] (fnil conj []) [(read-string p) v]))]
   ["-e" "--set-env KEY=VAL"     "Add KEY => VAL to project env map."
    :assoc-fn #(let [[k v] (string/split %3 #"=" 2)]
                 (update-in %1 [%2] (fnil assoc {}) (keyword k) v))]
   ["-h" "--help"                "Print basic usage and help info."]
   ["-P" "--no-profile"          "Skip loading of profile.boot script."]
   ["-s" "--src-paths PATH"      "Add PATH to set of source directories."
    :assoc-fn #(update-in %1 [%2] (fnil conj #{}) %3)]
   ["-t" "--tgt-path PATH"       "Set the target directory to PATH."]
   ["-V" "--version"             "Print boot version info."]])

(defn- parse-cli-opts [args]
  ((juxt :errors :options :arguments)
   (cli/parse-opts args cli-opts :in-order true)))

(defn- with-comments [tag forms]
  (concat
    [`(comment ~(format "start %s" tag))]
    forms
    [`(comment ~(format "end %s" tag))]))

(defn emit [argv userscript bootscript]
  `(~'(ns boot.user
        (:require
         [boot.task.built-in :refer :all]
         [boot.repl :refer :all]
         [boot.core :refer :all]
         [boot.util :refer :all :exclude [pp]]))
    ~@(when userscript (with-comments "profile" userscript))
    ~@(with-comments "boot script" bootscript)
    (if-let [main# (resolve 'boot.user/-main)]
      (main# ~@argv)
      (core/run-tasks (core/construct-tasks ~@(or (seq argv) ["boot.task.built-in/help"]))))))

(defn -main [[arg0 & args :as args*]]
  (let [dotboot?           #(.endsWith (.getName (io/file %)) ".boot")
        script?            #(when (and % (.isFile (io/file %)) (dotboot? %)) %)
        [errs opts args**] (parse-cli-opts (if (script? arg0) args args*))]
    
    (when (seq errs)
      (util/exit-error
        (println (apply str (interpose "\n" errs)))))

    (binding [*out* (util/auto-flush *out*)
              *err* (util/auto-flush *err*)]
      (util/exit-ok
        (let [[arg0 & args :as args*] (concat (if (script? arg0) [arg0] []) args**)
              bootscript  (io/file "build.boot")
              userscript  (script? (io/file (boot.App/getBootDir) "profile.boot"))
              [arg0 args] (cond
                            (script? arg0)       [arg0 args]
                            (script? bootscript) [bootscript args*]
                            :else                [nil args*])
              boot?       (contains? #{nil bootscript} arg0)
              profile?    (not (:no-profile opts))
              bootforms   (some->> arg0 slurp util/read-string-all)
              userforms   (when profile? (some->> userscript slurp util/read-string-all))
              commented   (concat () userforms [`(comment "script")] bootforms)
              scriptforms (emit args userforms bootforms)
              scriptstr   (str (string/join "\n\n" (map util/pp-str scriptforms)) "\n")]

          (when (:boot-script opts) (util/exit-ok (print scriptstr)))
          (when (:version opts) (util/exit-ok (println boot-version)))

          (reset! (var-get #'core/gitignore) (git/make-gitignore-matcher))
          (reset! (var-get #'core/tmpregistry) (tmp/init! (tmp/registry (io/file ".boot" "tmp"))))

          (#'core/init!
            :boot-version boot-version
            :boot-options opts
            :default-task 'boot.task.built-in/help)

          (let [tmpd (core/mktmpdir! ::bootscript)
                file #(doto (apply io/file %&) io/make-parents)
                tmpf (.getPath (file tmpd "boot" "user.clj"))]
            (core/set-env! :boot-user-ns-file tmpf)
            (doseq [[k v] (:set-env opts)] (core/set-env! k v))
            (doseq [k [:src-paths :tgt-path :dependencies]]
              (when-let [v (opts k)] (core/set-env! k v)))
            (try (doto tmpf (spit scriptstr) (load-file))
                 (catch clojure.lang.Compiler$CompilerException cx
                   (let [l (.-line cx)
                         s (->> (io/file (.-source cx))
                             (file/relative-to (io/file "."))
                             .getPath)
                         c (.getCause cx)
                         m (.getMessage (or c cx))
                         x (or c cx)]
                     (throw (ex-info m {:line l :file s} x)))))))))))
