(ns boot.native
  (:import
    [java.util.jar JarFile])
  (:require
    [clojure.java.io :as io]))

(defn- ^{:boot/from :technomancy/leiningen}
  get-by-pattern
  "Gets a value from map m, but uses the keys as regex patterns, trying
  to match against k instead of doing an exact match."
  [m k]
  (m (first (drop-while #(nil? (re-find (re-pattern %) k))
                        (keys m)))))

(defn- ^{:boot/from :technomancy/leiningen}
  get-with-pattern-fallback
  "Gets a value from map m, but if it doesn't exist, fallback
  to use get-by-pattern."
  [m k]
  (let [exact-match (m k)]
    (if (nil? exact-match)
      (get-by-pattern m k)
      exact-match)))

(def ^{:private true :boot/from :technomancy/leiningen}
  native-names
  {"Mac OS X" :macosx
   "Windows"  :windows
   "Linux"    :linux
   "FreeBSD"  :freebsd
   "OpenBSD"  :openbsd
   "amd64"    :x86_64
   "x86_64"   :x86_64
   "x86"      :x86
   "i386"     :x86
   "arm"      :arm
   "SunOS"    :solaris
   "sparc"    :sparc
   "Darwin"   :macosx})

(defn ^{:boot/from :technomancy/leiningen}
  get-os
  "Returns a keyword naming the host OS."
  []
  (get-with-pattern-fallback native-names (System/getProperty "os.name")))

(defn ^{:boot/from :technomancy/leiningen}
  get-arch
  "Returns a keyword naming the host architecture"
  []
  (get-with-pattern-fallback native-names (System/getProperty "os.arch")))

(defn ^{:boot/from :technomancy/leiningen}
  platform-nullsink
  "Returns a file destination that will discard output."
  []
  (io/file (if (= :windows (get-os)) "NUL" "/dev/null")))

(defn extract-native
  [jar-path out-dir]
  (let [jar    (JarFile. jar-path)
        prefix (format "native/%s/%s/" (get-os) (get-arch))]
    (when (.getEntry jar prefix)
      (doseq [e (->> (.entries jar)
                     enumeration-seq
                     (map (memfn getName))
                     (filter #(.startsWith % prefix)))]
        ))))
