(ns boot.jar
  (:require
   [clojure.java.io :as io]
   [boot.pod        :as pod]
   [boot.file       :as file]
   [boot.util       :as util])
  (:import
   [java.io File]
   [java.util.jar JarEntry JarOutputStream Manifest Attributes$Name]))

(def dfl-attr
  {"Created-By"  "Tailrecursion Boot Build Tool"
   "Built-By"    (System/getProperty "user.name")
   "Build-Jdk"   (System/getProperty "java.version")})

(defn create-manifest [main ext-attrs]
  (let [extra-attr  (merge-with into dfl-attr ext-attrs)
        manifest    (Manifest.) 
        attributes  (.getMainAttributes manifest)]
    (.put attributes Attributes$Name/MANIFEST_VERSION "1.0")
    (when-let [m (and main (.replaceAll (str main) "-" "_"))]
      (.put attributes Attributes$Name/MAIN_CLASS m))
    (doseq [[k v] extra-attr] (.put attributes (Attributes$Name. k) v))
    manifest))

(defn- write! [^JarOutputStream stream ^File file]
  (let [buf (byte-array 1024)]
    (with-open [in (io/input-stream file)]
      (loop [n (.read in buf)]
        (when-not (= -1 n)
          (.write stream buf 0 n)
          (recur (.read in buf)))))))

(defn spit-jar! [jarpath files attr main]
  (let [manifest  (create-manifest main attr)
        jarfile   (io/file jarpath)]
    (io/make-parents jarfile)
    (pod/info "Writing %s...\n" (.getName jarfile))
    (with-open [s (JarOutputStream. (io/output-stream jarfile) manifest)]
      (doseq [[jarpath srcpath] files :let [f (io/file srcpath)]]
        (pod/info "Adding %s...\n" jarpath)
        (let [entry (doto (JarEntry. jarpath) (.setTime (.lastModified f)))]
          (doto s (.putNextEntry entry) (write! f) .closeEntry))))))
