(ns util.jdbc
  (:use util.uuid)
  (:use util.storage)
  (:use [clojure.java.io :as io])
  (:require [clj-json [core :as json]]))

;;
;; Each struct is assumed to be limited in size to 64k of serialized JSON
;;

(defn jdbc-structstore
  "Return an IStructStore instance connected to the JDBC system specified by the settings. Possible keys are :classname, :subprotocol, :subname, :user, and :password"
  [settings]
  (let [url (str "jdbc:" (:subprotocol settings) ":" (:subname settings))
        keep (or (:keep settings) 1)]
    (sql/with-connection settings
      (try
        (and (sql/with-query-results rows ["SELECT * FROM asset limit 1"] (first rows) true))
        (catch Exception e
          (sql/create-table "asset" [:key "varchar(128)" "PRIMARY KEY"] [:data "VARCHAR(65536)"]))))

    (defn scan-results [rows limit]
      (let [results (apply array-map (flatten (map (fn [row] (let [k (:key row)] [k (json->struct (:data row))])) rows)))
            [last-key last-struct] (last results)
            base (if (and last-key (= (count results) limit)) {:more last-key} {})]
        (assoc base :structs results)))

    (proxy [util.storage.IStructStore] []

      (^clojure.lang.IPersistentMap scan [^String prefix ^String skip ^Number limit]
        (sql/with-connection settings
          (let [lim (if limit (str " limit " limit) "")]
            (if prefix
              (let [stop (str (.substring prefix 0 (dec (count prefix))) (char (+ 1 (int (last prefix)))))]
                (if skip
                  (sql/with-query-results rows [(str "SELECT key, data FROM asset where key > ? and key < ?" lim) skip stop]
                    (scan-results rows limit))
                  (sql/with-query-results rows [(str "SELECT key, data FROM asset where key >= ? and key < ?" lim) prefix stop]
                    (scan-results rows limit))))
              (if skip
                (sql/with-query-results rows [(str "SELECT key, data FROM asset where key > ?" lim) skip]
                  (scan-results rows limit))
                (sql/with-query-results rows [(str "SELECT key, data FROM asset" lim)]
                  (scan-results rows limit)))))))

      (^clojure.lang.IPersistentMap put [^String key ^clojure.lang.IPersistentMap item]
        (sql/with-connection settings
          (let [s (struct->json item)]
            (sql/update-or-insert-values "asset" ["key=?" key] {:key key :data s}) ;;the return value of this is inconsistent, not usable.
            true)))

      (^clojure.lang.ISeq multiget [^clojure.lang.ISeq keys]
        (sql/with-connection settings
          (sql/with-query-results rows [(str "SELECT data FROM asset where key in ?;") keys]
			(doall (map (fn [row]
				(json->struct (:data row))) rows)))))

      (^clojure.lang.ISeq multiput [^clojure.lang.ISeq keys ^clojure.lang.ISeq items]
        (sql/with-connection settings
			(doall (map (fn [key item]
          	   (let [s (struct->json item)]
            	  (sql/update-or-insert-values "asset" ["key=?" key] {:key key :data s})))
						keys items))))

      (^clojure.lang.IPersistentMap get [^String key]
        (sql/with-connection settings
          (sql/with-query-results rows [(str "SELECT data FROM asset where key=?;") key]
            (json->struct (:data (first rows))))))

      (^boolean delete [^String key]
        (sql/with-connection settings
          (let [n (first (sql/delete-rows "asset" ["key=?" key]))]
            (not (= 0 n))))))))

;; ideally, a "with-structstore" macro that keeps the DB open would be helpful. Use a state variable that the jdbc impl generates, somehow.

(defn h2-structstore
  "Return an IStructStore instance connected to an H2 database with the given name. It is created if it doesn't exist"
  [name]
  (jdbc-structstore {:classname "org.h2.Driver" :subprotocol "h2" :subname name}))

(defn sqlite-structstore
  "Return an IStructStore instance connected to a sqlite database with the given name. It is created if it doesn't exist"
  [name]
  (jdbc-structstore {:classname "org.sqlite.JDBC" :subprotocol "sqlite" :subname (str name ".sqlite")}))

