(ns util.zookeeper
  "deploys a single node Zookeeper instance in EC2"
  (:use util.supervise)
  (:use util.ec2))

(def ZK_NAME "zookeeper")
(def ZK_VERSION "3.3.5")
(def ZK_MACHINE_TYPE "m1.small")
(def ZK_LOG_FILE "zk-install.log")

(defn generate-zoo-config [dir]
  (format "## Generated by proto.zookeeper
tickTime=2000
initLimit=10
syncLimit=5
dataDir=%s
clientPort=2181
" dir))

(defn generate-zoo-run []
  (format "#!/bin/bash
start-foreground
export ZOOBINDIR=\".\"
if [ \"x$JMXLOCALONLY\" = \"x\" ]
then
    JMXLOCALONLY=false
fi
if [ \"x$JMXDISABLE\" = \"x\" ]
then
    echo \"JMX enabled by default\"
    # for some reason these two options are necessary on jdk6 on Ubuntu
    #   accord to the docs they are not necessary, but otw jconsole cannot
    #   do a local attach
    ZOOMAIN=\"-Dcom.sun.management.jmxremote -Dcom.sun.management.jmxremote.local.only=$JMXLOCALONLY org.apache.zookeeper.server.quorum.QuorumPeerMain\"
else
    echo \"JMX disabled by user request\"
    ZOOMAIN=\"org.apache.zookeeper.server.quorum.QuorumPeerMain\"
fi
if [ \"x$2\" != \"x\" ]
then
    ZOOCFG=\"$ZOOCFGDIR/$2\"
fi
cd bin && . ./zkEnv.sh && java  \"-Dzookeeper.log.dir=${ZOO_LOG_DIR}\" \"-Dzookeeper.root.logger=${ZOO_LOG4J_PROP}\" -cp \"$CLASSPATH\" $JVMFLAGS $ZOOMAIN \"$ZOOCFG\"
"))

(defn init-zookeeper [m & {:keys [capture data-dir] :or {capture ZK_LOG_FILE}}]
  (when m
    (deploy-supervise m) ;;make sure we have supervise deployed already
    (let [zk (format "zookeeper-%s" ZK_VERSION)
          home (.trim (command m "pwd" :capture :string))
          zk-home (str home "/zookeeper")
          datadir (or data-dir (str zk-home "/data"))
          zoo-conf (generate-zoo-config datadir)]
      (command m (format "curl -s -o %s.tar.gz http://www.apache.org/dist/zookeeper/%s/%s.tar.gz && echo %s downloaded" zk zk zk zk) :capture capture)
      (command m (format "tar xfz %s.tar.gz && ln -s %s zookeeper" zk zk) :capture capture)
      (command m (format "mkdir -p %s && echo mkdir -p %s" datadir datadir) :capture capture)
      (write-remote-file m zoo-conf (format "%s/conf/zoo.cfg" zk))
      (supervise m "zookeeper" (format "%s/bin/zkServer.sh start-foreground %s/conf/zoo.cfg" zk-home zk-home))
      m)))

(defn start-zookeeper [& {:keys [capture data-dir] :or {capture ZK_LOG_FILE}}]
  (let [m (create-machine ZK_NAME :type "m1.large")]
    (init-zookeeper m :capture capture :data-dir data-dir)))

(defn zookeeper-instance []
  (first (machines ZK_NAME)))

(defn zookeeper-deployed? []
  (if (zookeeper-instance) true false))

(defn deploy-zookeeper [& {:keys [capture data-dir] :or {capture ZK_LOG_FILE}}]
  (let [m (zookeeper-instance)]
    (or m (start-zookeeper :capture capture :data-dir data-dir))))

(defn stop-zookeeper []
  (destroy-cluster ZK_NAME)
  (println ZK_NAME "stopped"))
