(ns brave.swords
  (:require [clojure.string :as str]
            [clojure.tools.reader.edn :as edn]
            [goog.events :as events]
            [goog.string :as gstr]
            [goog.net.XhrIoPool  :as gxhr-pool]
            [goog.Uri.QueryData  :as gquery-data]
            [goog.structs        :as gstructs]
            [goog.net.EventType]
            [goog.net.ErrorCode])
  (:import [goog.events EventType]
           [goog.async Throttle Debouncer]
           ))

(defn timestamp [] (.getTime (js/Date.)))
(defn nil->str [x] (if (or (undefined? x) (nil? x)) "nil" x))
(defn format* [fmt args] (let [fmt  (or fmt "") args (mapv nil->str args)] (apply gstr/format fmt args)))
(defn formato [fmt & args] (format* fmt args))
(defn drop-nth [n coll] (keep-indexed #(if (not= %1 n) %2) coll))
(defn deaccent [input]
  (reduce (fn [s [pat repl]]
            (clojure.string/replace s pat repl))
          input
          [[#"[éèêë]" "e"]
           [#"[á]"    "a"]
           [#"[űúü]"  "u"]
           [#"[őöó]"  "o"]
           [#"[í]"    "i"]
           ]))

(defn disposable->function [disposable listener interval]
  (let [disposable-instance (disposable. listener interval)]
    (fn [& args]
      (.apply (.-fire disposable-instance) disposable-instance (to-array args)))))
(defn throttle [listener interval] (disposable->function Throttle listener interval))
(defn debounce [listener interval] (disposable->function Debouncer listener interval))

(defn safe-read 
  [value] (try (edn/read-string value) (catch js/Error e e)))

(defn extension [s] (-> s (clojure.string/split #"\.") last))

(defn event->value 
  [event]
  (let [
        value (-> event .-target .-value)
        readed-value (try (edn/read-string value) (catch js/Error e value));(edn/read-string {:eof value} value) 
        value 
        (cond 
          (keyword? readed-value) readed-value
          (boolean? readed-value) readed-value
          (integer? readed-value) readed-value
          (double? readed-value) readed-value
          (map? readed-value) readed-value 
          (vector? readed-value) readed-value 
          :else readed-value)] value))

(defn search [in filter-string]
  (if (or (= filter-string "") (= filter-string nil))  
    true (clojure.string/includes? (deaccent (clojure.string/lower-case in)) filter-string)))

(defn fancy [sample]
 (loop [x [0      []    0    false      4        ]  ]
  (let [  [index  coll  lvl  in-a-str?  line-type] x
        exist (< index (count sample))
        c (when exist (nth sample index))
        last-c (when-not (= index 0) (nth sample (dec index)))
        next-c (when-not (> index (-> sample count (- 2))) (nth sample (inc index)))
        prev? (fn [lc nc] (and (= last-c lc) (= c nc)))
        line-type (cond
                    (prev? \[ \{) 1
                    (prev? \} \]) 2
                    :else line-type)
        this-is-a-set (and (= last-c \#) (= c \{))
        lvl (cond
              (not last-c) 0
              this-is-a-set lvl ;To avoid splitting sets
              :else ((case c \{ + \} - \[ + \] - \( + \) - (fn [x y] x)) lvl line-type))
        c (let [new-line (apply conj [\newline] (repeat lvl \space))]
            (if-not (or in-a-str? this-is-a-set (not last-c))
              (case c
                \newline ""
                \, new-line 
                \{ (if (or (= last-c \[) (= next-c \})) c (conj new-line c))
                \[ (if (or (#{\[ \{} last-c) (= next-c \])) c (conj new-line c))
                \( (if (= last-c \[) c (conj new-line c))
                c)
              ;(case c
              ;  \n (if (and in-a-str? (= last-c \\)) (conj new-line c) c)
                c;)
              )
            )
        coll (if (vector? c) (apply conj coll c) (conj coll c))
        in-a-str? (if (and (= c \") (not= last-c \\)) (if in-a-str? false true) in-a-str?)
        return [(inc index) coll lvl in-a-str? line-type]]
  (if c (recur return)
    (apply str coll)))))

(defn deep-merge [v & vs]
  (letfn [(rec-merge [v1 v2]
            (if (and (map? v1) (map? v2))
              (merge-with deep-merge v1 v2)
              v2))]
    (when (some identity vs)
      (reduce #(rec-merge %1 %2) v vs))))

(defn round [x & {p :precision}] (if p (let [scale (Math/pow 10 p)] (-> x (* scale) Math/round (/ scale))) (Math/round x)))
(defn map-vals "map a function on values" [m f] (into {} (for [[k v] m] [k (f v)])))
(defn map-keys "map a function on keys" [f m] (if (nil? m) {} (reduce-kv (fn [m k v] (assoc m (f k) v)) {} m)))
(defn redirect! [loc] (set! (.-location js/window) loc))
(defn title! [t] (set! (. js/document -title) t))
(defn by-id "Short-hand for document.getElementById(id)" [id] (.getElementById js/document id))
(defn val-by-id [id] (if id (.-value (by-id id)) (js/console.log (str "No id:"id))))
(defn set-val! [id update] (if id (set! (.-value (by-id id)) update) (js/console.log (str "Can't set: "id))))
(defn set-display! [id update] (set! (.-style.display (by-id id)) update))
(defn set-html! [id s] (set! (.-innerHTML (by-id id)) s))

(defn jsx->clj [x] (into {} (for [k (.keys js/Object x)] [k (aget x k)])))
(defn obj->clj
  [obj]
  (-> (fn [result key]
        (let [v (aget obj key)]
          (if (= "function" (goog/typeOf v))
            result
            (assoc result key v))))
      (reduce {} (.getKeys goog/object obj))))

(defn user-agent [] (-> js/navigator .-userAgent))
(defn domain [] (-> js/document .-domain))
(defn title [] (-> js/document .-title))
(defn referrer [] (-> js/document .-referrer))
(defn prev-sites [] (-> js/history .-length))
(defn client-time [] (.Date js/window))
(defn timezone [] (/ (.getTimezoneOffset (client-time)) 60))
(defn protocol [] (-> js/window .-location .-protocol))
(defn host [] (-> js/window .-location .-host))
(defn host-name [] (-> js/window .-location .-hostname))
(defn url-hash [] (-> js/window .-location .-hash))
(defn url-path [] (-> js/window .-location .-pathname))
(defn browser [] (-> js/navigator .-appCodeName))
(defn browser-name [] (-> js/navigator .-appName))
(defn browser-engine [] (-> js/navigator .-product))
(defn browser-version [] (-> js/navigator .-appVersion))
(defn browser-plugins [] (obj->clj (-> js/navigator .-plugins)))
(defn browser-mimes [] (obj->clj (-> js/navigator .-mimeTypes)))
(defn browser-lang [] (-> js/navigator .-language))
(defn browser-languages [] (js->clj (-> js/navigator .-languages)))
(defn charset [] (-> js/document .-charset))
(defn online? [] (-> js/navigator .-onLine))
(defn connection [] (obj->clj (-> js/navigator .-connection)))
(defn os [] (-> js/navigator .-platform))
(defn cpu-cores [] (-> js/navigator .-hardwareConcurrency))
(defn performance-timing [] (obj->clj (-> js/window .-performance .-timing)))
(defn performance-memory [] (obj->clj (-> js/window .-performance .-memory)))
(defn media-devices [] (obj->clj (-> js/navigator .-mediaDevices)))
(defn cookies? [] (-> js/navigator .-cookieEnabled))
(defn cookies [] (-> js/document .-cookie))
(defn local-storage [] (js/localStorage))
(defn viewport [] (obj->clj (-> js/document .-visualViewport)))
(defn screen-width [] (-> js/screen .-width))
(defn screen-height [] (-> js/screen .-height))
(defn available-width [] (-> js/screen .-availWidth))
(defn mobile? [] (< (available-width) 768))
(defn tablet? [] (= (available-width) 768))
(defn pc? [] (> (available-width) 768))
(defn available-height [] (-> js/screen .-availHeight))
(defn color-depth [] (-> js/screen .-colorDepth))
(defn pixel-depth [] (-> js/screen .-pixelDepth))

(defn window-location "Returns `js/window`'s current location as a map."
  []
  (when-let [js-win (when (exists? js/window) js/window)]
    (when-let [loc (.-location js-win)]
      {:href     (.-href     loc) ; "http://www.example.org:80/foo/bar?q=baz#bang"
       :protocol (.-protocol loc) ; "http:" ; Note the :
       :hostname (.-hostname loc) ; "example.org"
       :host     (.-host     loc) ; "example.org:80"
       :pathname (.-pathname loc) ; "/foo/bar"
       :search   (.-search   loc) ; "?q=baz"
       :hash     (.-hash     loc) ; "#bang"
       })))

(defn on-click [id event]
  (when-let [target-el (by-id id)]
  (.addEventListener target-el "click" event)))

(defn on-key-press [id event]
  (when-let [target-el (by-id id)]
  (.addEventListener target-el "keydown" event)))

(defn dom-ready [event]
  (.addEventListener js/document "DOMContentLoaded" event))

(defn reduce-kvs
   "Like `reduce-kv` but takes a flat sequence of kv pairs."
  [rf init kvs]
  (transduce (partition-all 2)
    (completing (fn [acc [k v]] (rf acc k v))) init kvs))

(defn assoc-some "Assocs each kv iff its value is not nil."
    ([m k v      ] (if (nil? v) (if (nil? m) {} m) (assoc m k v)))
    ([m k v & kvs]
     (reduce-kvs
       (fn [m k v] (if (nil? v) m (assoc m k v)))
       (assoc-some m k v)
       kvs))

    ([m kvs]
     (reduce-kv
       (fn [m k v] (if (nil? v) m (assoc m k v)))
       (if (nil? m) {} m)
       kvs)))

(defn rsome      [pred coll] (reduce    (fn [acc in]  (when-let [p (pred in)]  (reduced p)))     nil coll))

(defn str-starts-with? [s substr] (zero? (.indexOf s substr)))
(defn str-contains? [s substr] (not= -1 (.indexOf s substr)))

(defn read-edn
  ([     s] (read-edn nil s))
  ([opts s]
   (if (or (nil? s) (identical? s ""))
     nil
     (if-not (string? s)
       (throw (ex-info "`read-edn` attempt against non-nil, non-string arg" {:given s :type (type s)}))
       (edn/read-string opts s)))))

(defn pr-edn
  "Prints arg to an edn string readable with `read-edn`."
  ([      x] (pr-edn nil x))
  ([_opts x] (binding [*print-level* nil, *print-length* nil] (pr-str x))))


(def xhr-pool_ (delay (goog.net.XhrIoPool.)))
  
(defn- get-pooled-xhr! [] (let [result (.getObject @xhr-pool_)] (if (undefined? result) nil result)))

(defn ajax-lite
 "(ajax-lite \"/my-post-route\"
    {:method     :post
     :params     {:username \"Rich Hickey\" :type \"Awesome\"}
     :headers    {\"Foo\" \"Bar\"}
     :resp-type  :text
     :timeout-ms 7000
     :with-credentials? false ; Enable if using CORS (requires xhr v2+)
    }
    (fn async-callback-fn [resp-map]
      (let [{:keys [success? ?status ?error ?content ?content-type]} resp-map]
        ;; ?status - e/o #{nil 200 404 ...}, non-nil iff server responded
        ;; ?error  - e/o #{nil <http-error-status-code> <exception> :timeout
                           :abort :http-error :exception :xhr-pool-depleted}
        (js/alert (str \"Ajax response: \" resp-map)))))
  [1] Ref. https://developers.google.com/closure/library/docs/xhrio"

  [uri {:keys [method params headers timeout-ms resp-type with-credentials?] :as opts
        :or   {method :get timeout-ms 10000 resp-type :auto}}
   callback-fn]

  (if-let [xhr (get-pooled-xhr!)]
    (let [
            xhr-method (case method :get "GET" :post "POST" :put "PUT")

            [xhr-uri xhr-?data]
            (let [url-encode 
                   (fn url-encode
                     ([params] (when (seq params) (-> params clj->js gstructs/Map. gquery-data/createFromMap .toString)))
                     ([uri params] (let [qstr (url-encode params) uri-with-query (if (str/blank? qstr) uri (str uri "?" qstr))] [uri-with-query nil]))) 
                  adaptive-encode
                   (fn [uri params]
                     (cond
                       (and (exists? js/FormData) (instance? js/FormData params)) [uri params]
                       (and (exists? js/FormData) (exists? js/File) (rsome (fn [x] (instance? js/File x)) (vals params)))
                       (let [form-data (js/FormData.)] (doseq [[k v] params] (.append form-data (name k) v)) [uri form-data])
                       :else [uri (url-encode params)]))]
               (case method
                 :get  (url-encode      uri params)
                 :post (adaptive-encode uri params)
                 :put  (adaptive-encode uri params)))

            xhr-headers
            (let [headers (map-keys #(str/lower-case (name %)) headers)
                  headers (assoc-some headers "x-requested-with"
                                 (get headers "x-requested-with" "XMLHTTPRequest"))]
              (clj->js headers))

            ?progress-listener
            (when-let [pf (:progress-fn opts)]
              (.setProgressEventsEnabled xhr true)
              (events/listen xhr goog.net.EventType/PROGRESS
                              (fn [ev]
                                (let [length-computable? (.-lengthComputable ev)
                                      loaded (.-loaded ev)
                                      total  (.-total  ev)
                                      ?ratio (when (and length-computable? (not= total 0))
                                               (/ loaded total))]
                                  (pf
                                   {:?ratio ?ratio
                                    :length-computable? length-computable?
                                    :loaded loaded
                                    :total  total
                                    :ev     ev})))))]

        (doto xhr
          (events/listenOnce goog.net.EventType/READY
            (fn [_] (.releaseObject @xhr-pool_ xhr)))

          (events/listenOnce goog.net.EventType/COMPLETE
            (fn wrapped-callback-fn [resp]
              (let [success? (.isSuccess xhr) ; true iff no error or timeout
                    -status  (.getStatus xhr) ; -1, 200, etc.

                    [?status ?content-type ?content]
                    (when (not= -status -1) ; Got a response from server
                      (let [;; Case insensitive get:
                            ?content-type (.getResponseHeader xhr "content-type")
                            ?content
                            (let [resp-type
                                  (cond
                                    (not= resp-type :auto) resp-type
                                    (nil? ?content-type)   :text
                                    :else
                                    (let [cts (str/lower-case (str ?content-type))
                                          match? (fn [s] (str-contains? cts s))]
                                      (cond
                                        (match? "/edn")     :edn
                                        (match? "/json")    :json
                                        (match? "/xml")     :xml
                                        ;; (match? "/html") :text
                                        :else               :text)))]

                              (case resp-type
                                  :edn  (read-edn (.getResponseText xhr))
                                  :json           (.getResponseJson xhr)
                                  :xml            (.getResponseXml  xhr)
                                  :text           (.getResponseText xhr)
                                {:ajax/bad-response-type resp-type
                                 :ajax/resp-as-text (.getResponseText xhr)}))]

                        [-status ?content-type ?content]))]

                (when ?progress-listener
                  (events/unlistenByKey ?progress-listener))

                (callback-fn
                  {:raw-resp      resp
                   :xhr           xhr ; = (.-target resp)
                   :success?      success?
                   :?status       ?status
                   :?content-type ?content-type
                   :?content      ?content
                   :?error
                   (if success?
                     nil
                     (cond
                       ?status ?status ; Http error status code (e.g. 404)
                       :else
                       (get {goog.net.ErrorCode/NO_ERROR   nil
                             goog.net.ErrorCode/EXCEPTION  :exception
                             goog.net.ErrorCode/HTTP_ERROR :http-error
                             goog.net.ErrorCode/ABORT      :abort
                             goog.net.ErrorCode/TIMEOUT    :timeout}
                         (.getLastErrorCode xhr)
                         :unknown)))})))))

        (.setTimeoutInterval xhr (or timeout-ms 0)) ; nil = 0 = no timeout
        (when with-credentials?
          (.setWithCredentials xhr true)) ; Requires xhr v2+

        (.send xhr xhr-uri xhr-method xhr-?data xhr-headers)
        xhr)
    (do ; Pool failed to return an available xhr instance
      (callback-fn {:?error :xhr-pool-depleted})
      nil)))
