(ns brave.zente
     (:require
      [clojure.string :as str]
      [cljs.core.async :as async :refer (<! >! put! chan)]
      [brave.swords :as x]
      [brave.interfaces :as interfaces])
     (:require-macros
      [cljs.core.async.macros :as asyncm :refer (go go-loop)]
      ))

(defn as-?nblank [x] (when (string? x) (if (str/blank? x) nil x)))

(defn as-?qname  [x]
  (cond
    (string? x) x
    :else (let [n (name x)] (if-let [ns (namespace x)] (str ns "/" n) n))))

(defn- assoc-conj [m k v] (assoc m k (if-let [cur (get m k)] (if (vector? cur) (conj cur v) [cur v]) v)))

(defn url-decode [s] (js/decodeURIComponent s))
(defn url-encode [s] (-> (str s) (js/encodeURIComponent s) (str/replace "*" "%2A") (str/replace "'" "%27")))

(defn parse-query-params "Based on `ring-codec/form-decode`."
  [s & [keywordize? encoding]]
  (if (or (str/blank? s) (not (x/str-contains? s "=")))
    {}
    (let [;; For convenience (e.g. JavaScript win-loc :search)
          s (if (x/str-starts-with? s "?") (subs s 1) s)
          m (reduce
              (fn [m param]
                (if-let [[k v] (str/split param #"=" 2)]
                  (assoc-conj m (url-decode k) (url-decode v))
                  m))
              {}
              (str/split s #"&"))]
      (if-not keywordize?
        m
(x/map-keys keyword m)))))

(defn format-query-string [m]
  (let [param (fn [k v]  (str (url-encode (as-?qname k)) "="
                             (url-encode (or (as-?qname v) (str v)))))
        join  (fn [strs] (str/join "&" strs))]
    (if (empty? m)
      ""
      (join
        (for [[k v] m :when (some? v)]
          (if (sequential? v)
            (join (mapv (partial param k) (or (seq v) [""])))
(param k v)))))))

(defn merge-url-with-query-string [url m]
    (let [[url ?qstr] (str/split (str url) #"\?" 2)
                  qmap  (merge
                                          (when ?qstr (x/map-keys keyword (parse-query-params ?qstr)))
                                          (x/map-keys keyword m))
                  ?qstr (as-?nblank (format-query-string qmap))]
      (if-let [qstr ?qstr] (str url "?" qstr) url)))

(defn oget
  ([o k          ] (goog.object/get o k nil))
  ([o k not-found] (goog.object/get o k not-found)))

(defn as-?nempty-str [x]
       (when (string? x)
         (if (= x "") nil x)))

(defn get-substring
    ([s start] (as-?nempty-str (.substr s start))))

(defn uuid-str
    ([max-length] (get-substring (uuid-str) max-length))
    ([]
        (let [hex  (fn [] (.toString (rand-int 16) 16))
                       rhex (.toString (bit-or 0x8 (bit-and 0x3 (rand-int 16))) 16)]
               (str (hex) (hex) (hex) (hex)
                              (hex) (hex) (hex) (hex) "-"
                              (hex) (hex) (hex) (hex) "-"
                              "4"   (hex) (hex) (hex) "-"
                              rhex  (hex) (hex) (hex) "-"
                              (hex) (hex) (hex) (hex)
                              (hex) (hex) (hex) (hex)
                    (hex) (hex) (hex) (hex)))))

(defn- expected [expected x] {:expected expected :actual {:type (type x) :value x}})
(defn validate-event
  "Returns nil if given argument is a valid [ev-id ?ev-data] form. Otherwise
  returns a map of validation errors like `{:wrong-type {:expected _ :actual _}}`."
  [x]
  (cond
    (not (vector? x))        {:wrong-type   (expected :vector x)}
    (not (#{1 2} (count x))) {:wrong-length (expected #{1 2}  x)}
    :else
    (let [[ev-id _] x]
      (cond
        (not (keyword? ev-id))  {:wrong-id-type   (expected :keyword            ev-id)}
        (not (namespace ev-id)) {:unnamespaced-id (expected :namespaced-keyword ev-id)}
        :else nil))))

(defn assert-event
  [x]
  (when-let [errs (validate-event x)]
    (throw (ex-info "Invalid event" {:given x :errors errs}))))

(defn as-event [x]
  (if-let [errs (validate-event x)]
    [:chsk/bad-event x]
    x))

(defn- unpack "prefixed-pstr->[clj ?cb-uuid]"
  [packer prefixed-pstr]
  (let [wrapped? (x/str-starts-with? prefixed-pstr "+")
        pstr     (subs prefixed-pstr 1)
        clj
        (try
          (interfaces/unpack packer pstr)
          (catch :default t
            [:chsk/bad-package pstr]))

        [clj ?cb-uuid] (if wrapped? clj [clj nil])
        ?cb-uuid (if (= 0 ?cb-uuid) :ajax-cb ?cb-uuid)]
    
    [clj ?cb-uuid]))

(defn- pack "clj->prefixed-pstr"
  ([packer clj]
   (let [;; "-" prefix => Unwrapped (has no callback)
         pstr (str "-" (interfaces/pack packer clj))]
     pstr))

  ([packer clj ?cb-uuid]
   (let [;;; Keep wrapping as light as possible:
         ?cb-uuid    (if (= ?cb-uuid :ajax-cb) 0 ?cb-uuid)
         wrapped-clj (if ?cb-uuid [clj ?cb-uuid] [clj])
         ;; "+" prefix => Wrapped (has callback)
         pstr (str "+" (interfaces/pack packer wrapped-clj))]
     
     pstr)))

(deftype EdnPacker []
  interfaces/IPacker
  (pack   [_ x] (x/pr-edn   x))
  (unpack [_ s] (x/read-edn s)))

(def ^:private default-edn-packer (EdnPacker.))

(declare ^:private default-client-side-ajax-timeout-ms)

(defprotocol IChSocket
     (-chsk-connect!    [chsk])
     (-chsk-disconnect! [chsk reason])
     (-chsk-reconnect!  [chsk])
     (-chsk-send!       [chsk ev opts]))

(defn chsk-connect!    [chsk] (-chsk-connect!    chsk))
(defn chsk-disconnect! [chsk] (-chsk-disconnect! chsk :requested-disconnect))
(defn chsk-reconnect! [chsk] (-chsk-reconnect! chsk))
(def chsk-destroy! "Deprecated" chsk-disconnect!)

(defn chsk-send!
     ([chsk ev] (chsk-send! chsk ev {}))
     ([chsk ev ?timeout-ms ?cb] (chsk-send! chsk ev {:timeout-ms ?timeout-ms
                                                     :cb         ?cb}))
     ([chsk ev opts] (-chsk-send! chsk ev opts)))

(defn- chsk-send->closed! [?cb-fn]
     (when ?cb-fn (?cb-fn :chsk/closed))
     false)

(defn- pull-unused-cb-fn! [cbs-waiting_ ?cb-uuid]
     (when-let [cb-uuid ?cb-uuid] (swap! cbs-waiting_ update-in [cb-uuid] (fn [?f] [:swap/dissoc ?f]))))

(defn- swap-chsk-state!
     [chsk f]
     (let [old-state (:state_ chsk)
           _ (swap! (:state_ chsk) f)
           _ (swap! (:state_ chsk) #(if (:first-open? %) (assoc % :first-open? false)))
           _ (swap! (:state_ chsk) #(if (:open? %) (dissoc % :udt-next-reconnect)))]
       (when (not= old-state (:state_ chsk))
         (put! (get-in chsk [:chs :state]) [:chsk/state [old-state (:state_ chsk)]]))))

(defn- chsk-state->closed [state reason]
     (if (or (:open? state) (not= reason :unexpected))
       (-> state
           (dissoc :udt-next-reconnect)
           (assoc
             :open? false
             :last-close {:udt (.getTime (js/Date.)) :reason reason}))
       state))

(defn- cb-chan-as-fn
     "Experimental, undocumented. Allows a core.async channel to be provided
     instead of a cb-fn. The channel will receive values of form
     [<event-id>.cb <reply>]."
     [?cb ev]
     (if (or (nil? ?cb) (ifn? ?cb))
       ?cb
       (do
         (assert-event ev)
         (let [[ev-id _] ev
               cb-ch ?cb]
           (fn [reply]
             (put! cb-ch
               [(keyword (str (namespace ev-id)"/"(name ev-id)".cb"))
                reply]))))))

(defn- receive-buffered-evs! [chs clj]
       (doseq [ev clj]
         (assert-event ev)
         ;; Should never receive :chsk/* events from server here:
         (let [[id] ev] (assert (not= (namespace id) "chsk")))
         (put! (:<server chs) ev)))

(defn- handshake? [x]
     (and (vector? x) 
          (let [[x1] x] (= x1 :chsk/handshake))))

(defn- receive-handshake! [chsk-type chsk clj]
     (let [[_ [?uid _ ?handshake-data]] clj
           {:keys [chs ever-opened?_]} chsk
           first-handshake? (compare-and-set! ever-opened?_ false true)
           new-state
           {:type           chsk-type ; :auto -> e/o #{:ws :ajax}, etc.
            :open?          true
            :ever-opened?   true
            :uid            ?uid
            :handshake-data ?handshake-data
            :first-open?    first-handshake?}

           handshake-ev
           [:chsk/handshake
            [?uid nil ?handshake-data first-handshake?]]]

       (assert-event handshake-ev)

       (swap-chsk-state! chsk #(merge % new-state))
       (put! (:internal chs) handshake-ev)

       :handled))

(defrecord ChWebSocket
     ;; WebSocket-only IChSocket implementation
     ;; Handles (re)connections, cbs, etc.

     [client-id chs params packer url ws-kalive-ms
      state_ ; {:type _ :open? _ :uid _ :csrf-token _ ...}
      instance-handle_ retry-count_ ever-opened?_
      backoff-ms-fn ; (fn [nattempt]) -> msecs
      cbs-waiting_ ; {<cb-uuid> <fn> ...}
      socket_
      udt-last-comms_]

     IChSocket
     (-chsk-disconnect! [chsk reason]
       (reset! instance-handle_ nil) ; Disable auto retry
       (swap-chsk-state! chsk #(chsk-state->closed % reason))
       (when-let [s @socket_] (.close s 1000 "CLOSE_NORMAL")))

     (-chsk-reconnect! [chsk]
       (-chsk-disconnect! chsk :requested-reconnect)
       (-chsk-connect!    chsk))

     (-chsk-send! [chsk ev opts]
       (let [{?timeout-ms :timeout-ms ?cb :cb :keys [flush?]} opts
             _ (assert-event ev)
             ?cb-fn (cb-chan-as-fn ?cb ev)]
         (if-not (:open? @state_) ; Definitely closed
           (chsk-send->closed! ?cb-fn)

           ;; TODO Buffer before sending (but honor `:flush?`)
           (let [?cb-uuid (when ?cb-fn (uuid-str 6))
                 ppstr (pack packer ev ?cb-uuid)]

             (when-let [cb-uuid ?cb-uuid]
               (swap! cbs-waiting_ assoc-in [cb-uuid] ?cb-fn)
               (when-let [timeout-ms ?timeout-ms]
                 (go
                   (<! (async/timeout timeout-ms))
                   (when-let [cb-fn* (pull-unused-cb-fn! cbs-waiting_ ?cb-uuid)]
                     (cb-fn* :chsk/timeout)))))

             (try
               (.send @socket_ ppstr)
               (reset! udt-last-comms_ (.getTime (js/Date.)))
               :apparent-success
               (catch :default e
                 (when-let [cb-uuid ?cb-uuid]
                   (let [cb-fn* (or (pull-unused-cb-fn! cbs-waiting_ cb-uuid)
                                    ?cb-fn)]
                     (cb-fn* :chsk/error)))
                 false))))))

     (-chsk-connect! [chsk]
       (when-let [WebSocket
                  (or
                    (oget goog/global    "WebSocket")
                    (oget goog/global "MozWebSocket")
                    )]

         (let [instance-handle (reset! instance-handle_ (uuid-str))
               have-handle? (fn [] (= @instance-handle_ instance-handle))
               connect-fn
               (fn connect-fn []
                 (when (have-handle?)
                   (let [retry-fn
                         (fn [] ; Backoff then recur
                           (when (have-handle?)
                             (let [retry-count* (swap! retry-count_ inc)
                                   backoff-ms (backoff-ms-fn retry-count*)
                                   udt-next-reconnect (+ (.getTime (js/Date.)) backoff-ms)]
                               ;(js/console.log "Chsk is closed: will try reconnect attempt (%s) in %s ms")
                               (.setTimeout goog/global connect-fn backoff-ms)
                               (swap-chsk-state! chsk
                                 #(assoc % :udt-next-reconnect udt-next-reconnect)))))

                         ?socket
                         (try
                           (WebSocket.
                             (merge-url-with-query-string url
                               (merge params 
                                 {:client-id client-id})))

                           (catch :default e
                             nil))]

                     (if-not ?socket
                       (retry-fn) ; Couldn't even get a socket

                       (reset! socket_
                         (doto ?socket
                           (aset "onerror"
                             (fn [ws-ev]
                               (let [last-ws-error {:udt (.getTime (js/Date.)), :ev ws-ev}]
                                 (swap-chsk-state! chsk
                                   #(assoc % :last-ws-error last-ws-error)))))

                           (aset "onmessage" ; Nb receives both push & cb evs!
                             (fn [ws-ev]
                               (let [ppstr (oget ws-ev "data")

                                     ;; `clj` may/not satisfy `event?` since
                                     ;; we also receive cb replies here. This
                                     ;; is why we prefix pstrs to indicate
                                     ;; whether they're wrapped or not
                                     [clj ?cb-uuid] (unpack packer ppstr)]

                                 (reset! udt-last-comms_ (.getTime (js/Date.)))

                                 (or
                                   (when (handshake? clj)
                                     (receive-handshake! :ws chsk clj)
                                     (reset! retry-count_ 0)
                                     :handshake)

                                   (when (= clj :chsk/ws-ping)
                                     (put! (:<server chs) [:chsk/ws-ping])
                                     :noop)

                                   (if-let [cb-uuid ?cb-uuid]
                                     (if-let [cb-fn (pull-unused-cb-fn! cbs-waiting_
                                                      cb-uuid)]
                                       (cb-fn clj)
                                       )
                                     (let [buffered-evs clj]
                                       (receive-buffered-evs! chs buffered-evs)))))))

                           ;; Fires repeatedly (on each connection attempt) while
                           ;; server is down:
                           (aset "onclose"
                             (fn [ws-ev]
                               (let [clean? (oget ws-ev "wasClean")
                                     code   (oget ws-ev "code")
                                     reason (oget ws-ev "reason")
                                     last-ws-close
                                     {:udt    (.getTime (js/Date.))
                                      :ev     ws-ev
                                      :clean? clean?
                                      :code   code
                                      :reason reason}]

                                 ;; Firefox calls "onclose" while unloading,
                                 ;; Ref. http://goo.gl/G5BYbn:
                                 (if clean?
                                   (swap-chsk-state! chsk
                                       #(assoc % :last-ws-close last-ws-close))
                                   (do
                                     (swap-chsk-state! chsk
                                       #(assoc (chsk-state->closed % :unexpected)
                                          :last-ws-close last-ws-close))
                                     (retry-fn))))))))))))]

           (when-let [ms ws-kalive-ms]
             (go-loop []
               (let [udt-t0 @udt-last-comms_]
                 (<! (async/timeout ms))
                 (when (have-handle?)
                   (let [udt-t1 @udt-last-comms_]
                     (when (= udt-t0 udt-t1)
                       ;; Ref. issue #259:
                       ;; We've seen no send/recv activity on this
                       ;; conn w/in our kalive window so send a ping
                       ;; ->server (should auto-close conn if it's
                       ;; gone dead). The server generally sends pings so
                       ;; this should be rare. Mostly here to help clients
                       ;; identify conns that were suddenly dropped.

                       (-chsk-send! chsk [:chsk/ws-ping] {:flush? true})))
                   (recur)))))

           (reset! retry-count_ 0)
           (connect-fn)
           chsk))))

(defn- new-ChWebSocket [opts csrf-token]
     (map->ChWebSocket
       (merge
         {:state_ 
          (atom {:type :ws 
                 :open? false 
                 :ever-opened? false
                 :csrf-token csrf-token})
          :instance-handle_ (atom nil)
          :retry-count_     (atom 0)
          :ever-opened?_    (atom false)
          :cbs-waiting_     (atom {})
          :socket_          (atom nil)
          :udt-last-comms_  (atom nil)}
         opts)))

(def ^:private default-client-side-ajax-timeout-ms 60000)

(defrecord ChAjaxSocket
     ;; Ajax-only IChSocket implementation
     ;; Handles (re)polling, etc.

     [client-id chs params packer url state_
      instance-handle_ ever-opened?_
      backoff-ms-fn
      ajax-opts curr-xhr_]

     IChSocket
     (-chsk-disconnect! [chsk reason]
       (reset! instance-handle_ nil) ; Disable auto retry
       (swap-chsk-state! chsk #(chsk-state->closed % reason))
       (when-let [x @curr-xhr_] (.abort x)))

     (-chsk-reconnect! [chsk]
       (-chsk-disconnect! chsk :requested-reconnect)
       (-chsk-connect!    chsk))

     (-chsk-send! [chsk ev opts]
       (let [{?timeout-ms :timeout-ms ?cb :cb :keys [flush?]} opts
             _ (assert-event ev)
             ?cb-fn (cb-chan-as-fn ?cb ev)]
         (if-not (:open? @state_) ; Definitely closed
           (chsk-send->closed! ?cb-fn)

           ;; TODO Buffer before sending (but honor `:flush?`)
           (let [csrf-token (:csrf-token @state_)]
             (x/ajax-lite url
               (merge ajax-opts
                 {:method     :post
                  :timeout-ms (or ?timeout-ms (:timeout-ms ajax-opts)
                                  default-client-side-ajax-timeout-ms)
                  :resp-type  :text ; We'll do our own pstr decoding
                  :headers
                  (merge (:headers ajax-opts) ; 1st (don't clobber impl.):
                    {:X-CSRF-Token csrf-token})

                  :params
                  (let [ppstr (pack packer ev (when ?cb-fn :ajax-cb))]
                    (merge params ; 1st (don't clobber impl.):
                      {:udt        (.getTime (js/Date.)) ; Force uncached resp

                       ;; A duplicate of X-CSRF-Token for user's convenience
                       ;; and for back compatibility with earlier CSRF docs:
                       :csrf-token csrf-token

                       ;; Just for user's convenience here. non-lp-POSTs
                       ;; don't actually need a client-id for Sente's own
                       ;; implementation:
                       :client-id  client-id

                       :ppstr      ppstr}))})

               (fn ajax-cb [{:keys [?error ?content]}]
                 (if ?error
                   (if (= ?error :timeout)
                     (when ?cb-fn (?cb-fn :chsk/timeout))
                     (do
                       (swap-chsk-state! chsk
                         #(chsk-state->closed % :unexpected))
                       (when ?cb-fn (?cb-fn :chsk/error))))

                   (let [content ?content
                         resp-ppstr content
                         [resp-clj _] (unpack packer resp-ppstr)]
                     (when ?cb-fn (?cb-fn resp-clj))
                     (swap-chsk-state! chsk #(assoc % :open? true))))))

             :apparent-success))))

     (-chsk-connect! [chsk]
       (let [instance-handle (reset! instance-handle_ (uuid-str))
             have-handle? (fn [] (= @instance-handle_ instance-handle))
             poll-fn ; async-poll-for-update-fn
             (fn poll-fn [retry-count]
               (when (have-handle?)
                 (let [retry-fn
                       (fn [] ; Backoff then recur
                         (when (have-handle?)
                           (let [retry-count* (inc retry-count)
                                 backoff-ms (backoff-ms-fn retry-count*)
                                 udt-next-reconnect (+ (.getTime (js/Date.)) backoff-ms)]
                            ; (warnf "Chsk is closed: will try reconnect attempt (%s) in %s ms"
                            ;        retry-count* backoff-ms)
                             (.setTimeout goog/global
                               (fn [] (poll-fn retry-count*))
                               backoff-ms)
                             (swap-chsk-state! chsk
                               #(assoc % :udt-next-reconnect udt-next-reconnect)))))]

                   (reset! curr-xhr_
                     (x/ajax-lite url
                       (merge ajax-opts
                         {:method     :get ; :timeout-ms timeout-ms
                          :timeout-ms (or (:timeout-ms ajax-opts)
                                        default-client-side-ajax-timeout-ms)
                          :resp-type  :text ; Prefer to do our own pstr reading
                          :params
                          (merge
                            ;; Note that user params here are actually POST
                            ;; params for convenience. Contrast: WebSocket
                            ;; params sent as query params since there's no
                            ;; other choice there.
                            params ; 1st (don't clobber impl.):

                            {:udt       (.getTime (js/Date.)) ; Force uncached resp
                             :client-id client-id}

                            ;; A truthy :handshake? param will prompt server to
                            ;; reply immediately with a handshake response,
                            ;; letting us confirm that our client<->server comms
                            ;; are working:
                            (when-not (:open? @state_) {:handshake? true}))
                          :headers
                          (merge
                            (:headers ajax-opts) ; 1st (don't clobber impl.)
                            {:X-CSRF-Token (:csrf-token @state_)})})

                       (fn ajax-cb [{:keys [?error ?content]}]
                         (if ?error
                           (cond
                             (= ?error :timeout) (poll-fn 0)
                             ;; (= ?error :abort) ; Abort => intentional, not an error
                             :else
                             (do
                               (swap-chsk-state! chsk
                                 #(chsk-state->closed % :unexpected))
                               (retry-fn)))

                           ;; The Ajax long-poller is used only for events, never cbs:
                           (let [content ?content
                                 ppstr content
                                 [clj] (unpack packer ppstr)
                                 handshake? (handshake? clj)]

                             (when handshake?
                               (receive-handshake! :ajax chsk clj))

                             (swap-chsk-state! chsk #(assoc % :open? true))
                             (poll-fn 0) ; Repoll asap

                             (when-not handshake?
                               (or
                                 (when (= clj :chsk/timeout)
                                     (receive-buffered-evs! chs [[:debug/timeout]])
                                   :noop)

                                 (let [buffered-evs clj] ; An application reply
                                   (receive-buffered-evs! chs buffered-evs))))))))))))]

         (poll-fn 0)
         chsk)))

(defn- new-ChAjaxSocket [opts csrf-token]
     (map->ChAjaxSocket
       (merge
         {:state_           (atom {:type :ajax 
                                   :open? false 
                                   :ever-opened? false
                                   :csrf-token csrf-token})
          :instance-handle_ (atom nil)
          :ever-opened?_    (atom false)
          :curr-xhr_        (atom nil)}
         opts)))

(defrecord ChAutoSocket
     ;; Dynamic WebSocket/Ajax IChSocket implementation
     ;; Wraps a swappable ChWebSocket/ChAjaxSocket

     [ws-chsk-opts ajax-chsk-opts state_
      impl_ ; ChWebSocket or ChAjaxSocket
      ]

     IChSocket
     (-chsk-disconnect! [chsk reason]
       (when-let [impl @impl_]
         (-chsk-disconnect! impl reason)))

     ;; Possibly reset impl type:
     (-chsk-reconnect! [chsk]
       (when-let [impl @impl_]
         (-chsk-disconnect! impl :requested-reconnect)
         (-chsk-connect!    chsk)))

     (-chsk-send! [chsk ev opts]
       (if-let [impl @impl_]
         (-chsk-send! impl ev opts)
         (let [{?cb :cb} opts
               ?cb-fn (cb-chan-as-fn ?cb ev)]
           (chsk-send->closed! ?cb-fn))))

     (-chsk-connect! [chsk]
       ;; Starting with a simple downgrade-only strategy here as a proof of concept
       ;; TODO Later consider smarter downgrade or downgrade+upgrade strategies?
       (let [ajax-chsk-opts (assoc ajax-chsk-opts :state_ state_)
               ws-chsk-opts (assoc   ws-chsk-opts :state_ state_)

             ajax-conn!
             (fn []
               ;; Remove :auto->:ajax downgrade watch
               (remove-watch state_ :chsk/auto-ajax-downgrade)
               (-chsk-connect! (new-ChAjaxSocket ajax-chsk-opts (:csrf-token @state_))))

             ws-conn!
             (fn []
               ;; Configure :auto->:ajax downgrade watch
               (let [downgraded?_ (atom false)]
                 (add-watch state_ :chsk/auto-ajax-downgrade
                   (fn [_ _ old-state new-state]
                     (when-let [impl @impl_]
                       (when-let [ever-opened?_ (:ever-opened?_ impl)]
                         (when-not @ever-opened?_
                           (when (:last-ws-error new-state)
                             (when (compare-and-set! downgraded?_ false true)
                               (-chsk-disconnect! impl :downgrading-ws-to-ajax)
                               (reset! impl_ (ajax-conn!))))))))))

               (-chsk-connect! (new-ChWebSocket ws-chsk-opts (:csrf-token @state_))))]

         (reset! impl_ (or (ws-conn!) (ajax-conn!)))
         chsk)))

(defn- new-ChAutoSocket [opts csrf-token]
     (map->ChAutoSocket
       (merge
         {:state_ (atom {:type :auto 
                         :open? false 
                         :ever-opened? false
                         :csrf-token csrf-token})
          :impl_  (atom nil)}
         opts)))

(defn- get-chsk-url [protocol host path-url type]
     (let [protocol (case protocol :http "http:" :https "https:" protocol)
           protocol (case type
                      :ajax     protocol
                      :ws (case protocol "https:" "wss:" "http:" "ws:"))]
       (str protocol "//" host path-url)))

(defn exp-backoff "Returns binary exponential backoff value for n<=36."
  ([^long n-attempt] (exp-backoff n-attempt nil))
  ([^long n-attempt {:keys [min max factor] :or {factor 1000}}]
   (let [n (if (> n-attempt 36) 36 n-attempt) ; >2^36 excessive
         b (Math/pow 2 n)
         t (long (* (+ b ^double (rand b)) 0.5 (double factor)))
         t (long (if min (if (< t ^long min) min t) t))
         t (long (if max (if (> t ^long max) max t) t))] t)))

(defn make-channel-socket-client!
     [path ?csrf-token &
      [{:keys [type protocol host params recv-buf-or-n packer ws-kalive-ms
               client-id ajax-opts wrap-recv-evs? backoff-ms-fn]
        :as   opts
        :or   {type           :auto
               recv-buf-or-n  (async/sliding-buffer 2048) ; Mostly for buffered-evs
               packer         :edn
               client-id      (uuid-str)
               wrap-recv-evs? true
               backoff-ms-fn  exp-backoff
               ws-kalive-ms   20000}}

       _deprecated-more-opts]]

     (let [packer (if (= packer :edn) default-edn-packer packer)

           [ws-url ajax-url]
           (let [;; Not available with React Native, etc.:
                 win-loc  (x/window-location)
                 path     (or path (:pathname win-loc))
                 protocol (or protocol (:protocol win-loc) :http)
                 host     (or host     (:host     win-loc))]
                 [(get-chsk-url protocol host path :ws)
                  (get-chsk-url protocol host path :ajax)])

           private-chs
           {:internal (chan (async/sliding-buffer 128))
            :state    (chan (async/sliding-buffer 10))
            :<server
            (let [;; Nb must be >= max expected buffered-evs size:
                  buf (async/sliding-buffer 512)]
              (if wrap-recv-evs?
                (chan buf (map (fn [ev] [:chsk/recv ev])))
                (chan buf)))}

           common-chsk-opts
           {:client-id    client-id
            :chs          private-chs
            :params       params
            :packer       packer
            :ws-kalive-ms ws-kalive-ms}

           ws-chsk-opts
           (merge common-chsk-opts
             {:url           ws-url
              :backoff-ms-fn backoff-ms-fn})

           ajax-chsk-opts
           (merge common-chsk-opts
             {:url           ajax-url
              :ajax-opts     ajax-opts
              :backoff-ms-fn backoff-ms-fn})

           auto-chsk-opts
           {:ws-chsk-opts   ws-chsk-opts
            :ajax-chsk-opts ajax-chsk-opts}

           ?chsk
           (-chsk-connect!
             (case type
               :ws   (new-ChWebSocket    ws-chsk-opts ?csrf-token)
               :ajax (new-ChAjaxSocket ajax-chsk-opts ?csrf-token)
               :auto (new-ChAutoSocket auto-chsk-opts ?csrf-token)))]

       (if-let [chsk ?chsk]
         (let [chsk-state_ (:state_ chsk)
               internal-ch (:internal private-chs)
               send-fn (partial chsk-send! chsk)
               ev-ch
               (async/merge
                 [(:internal private-chs)
                  (:state    private-chs)
                  (:<server  private-chs)]
                 recv-buf-or-n)

               ev-msg-ch
               (async/chan 1
                 (map
                   (fn [ev]
                     (let [[ev-id ev-?data :as ev] (as-event ev)]
                       {;; Allow client to inject into router for handler:
                        :ch-recv internal-ch
                        :send-fn send-fn
                        :state   chsk-state_
                        :event   ev
                        :id      ev-id
                        :?data   ev-?data}))))]

           (async/pipe ev-ch ev-msg-ch)

           {:chsk    chsk
            :ch-recv ev-msg-ch
            :send-fn send-fn
            :state   (:state_ chsk)})

         ;(warnf "Failed to create channel socket")
         )))

(defn start-client-chsk-router!
  [ch-recv event-msg-handler]
  (let [ch-ctrl (chan)]
    (go-loop []
      (let [[v p] (async/alts! [ch-recv ch-ctrl])]
        (when-not (or (= p ch-ctrl) (nil? v))
          (let [{:as event-msg :keys [event]} v]
            (event-msg-handler event-msg)
            (recur)))))
    (fn stop! [] (async/close! ch-ctrl))))
