(ns plinth.login.password
  (:require
    [plinth.schema :as s]
    [plinth.table :as table]
    [clojure.tools.logging :as log]
    [clojurewerkz.scrypt.core :as sc]
    [com.stuartsierra.component :as component]))

(def PasswordLoginSchema
  (s/table :password-logins :username
    { :username (assoc s/Str :required true :unique true)
      :encrypted-password (s/required s/Str)
      :user-id s/Str }))

(def invalid-username   "Invalid username")
(def duplicate-username "Username already taken")
(def invalid-password   "Invalid username/password")

(defn- encrypt [password]
  (sc/encrypt password 16384 8 1))

(defn username-exists? [{:keys [table] :as login} username]
  (boolean (table/look-up table username)))

(defn find-user-id-by-username [{:keys [table] :as login} username]
  (:user-id (table/look-up table username)))

(defn create! [{:keys [table] :as login} user-id username password]
  (log/info "Creating login for" username)
  (when-not
    (table/insert! table
      { :user-id user-id
        :username username
        :encrypted-password (encrypt password) })
    (throw (ex-info duplicate-username {:username username}))))

(defn login! [{:keys [table] :as login} username password]
  (log/info "Logging in" username)
  (let [{:keys [user-id encrypted-password]} (table/look-up table username)]
    (if (sc/verify password encrypted-password)
      user-id
      (throw (ex-info invalid-password {:username username})))))

(defn reset-password! [{:keys [table] :as login} username password]
  (log/info "Resetting password for" username)
  (when-not (table/update! table username {:encrypted-password (encrypt password)})
    (throw (ex-info invalid-username {:username username}))))

(defn update-username! [{:keys [table] :as login} old-username new-username]
  (log/info "Updating username from" old-username "to" new-username)
  (when-not (table/update! table old-username {:username new-username})
    (throw (ex-info invalid-username {:old-username old-username :new-username new-username}))))

(defn delete! [{:keys [table] :as login} username]
  (log/info "Deleting login for username" username)
  (when-not (table/delete! table username)
    (throw (ex-info invalid-username {:username username}))))
