(ns plinth.schema.prismatic
  (:require
    [schema.core :as ps]
    [plinth.schema :as schema]
    [clojure.tools.logging :as log]))

(defn- upto [schema cnt]
  (into [] (repeat cnt (ps/optional schema (name (:name schema (:type schema :unknown)))))))

(defn- wrap-prismatic-schema [schema prismatic]
  (cond-> prismatic
    (:predicate schema)       (ps/both
                                (ps/pred
                                  (:predicate schema)
                                  (name (:name schema (:type schema :unknown)))))
    (not (:required schema))  (ps/maybe)
    (:cardinality schema)     (upto (:cardinality schema))))

(defmulti prismatic-schema #'schema/dispatch)
(defmethod prismatic-schema :any [schema] (wrap-prismatic-schema schema ps/Any))
(defmethod prismatic-schema :serial [schema] (wrap-prismatic-schema schema ps/Int))
(defmethod prismatic-schema :string [schema] (wrap-prismatic-schema schema ps/Str))
(defmethod prismatic-schema :number [schema] (wrap-prismatic-schema schema ps/Num))
(defmethod prismatic-schema :integer [schema] (wrap-prismatic-schema schema ps/Int))
(defmethod prismatic-schema :uuid [schema] (wrap-prismatic-schema schema ps/Uuid))
(defmethod prismatic-schema :instant [schema] (wrap-prismatic-schema schema ps/Inst))
; (defmethod prismatic-schema :time [schema] (wrap-prismatic-schema schema ps/Str))
; (defmethod prismatic-schema :date [schema] (wrap-prismatic-schema schema ps/Str))
(defmethod prismatic-schema :boolean [schema] (wrap-prismatic-schema schema ps/Bool))
(defmethod prismatic-schema :uri [schema] (wrap-prismatic-schema schema ps/Str))
(defmethod prismatic-schema :constant [schema] (wrap-prismatic-schema schema (ps/eq (:subschema schema))))
(defmethod prismatic-schema :enum [schema] (wrap-prismatic-schema schema (apply ps/enum (:subschema schema))))
(defmethod prismatic-schema :set [schema]
  (wrap-prismatic-schema schema (set (map prismatic-schema (:subschema schema)))))
(defmethod prismatic-schema :sequence [schema]
  (wrap-prismatic-schema schema
    (mapv #(ps/one (prismatic-schema %) (name (:name % (:type % :unknown)))) (:subschema schema))))
(defmethod prismatic-schema :value [schema]
  (wrap-prismatic-schema schema (prismatic-schema (:subschema schema))))
(defmethod prismatic-schema :clj-map [schema]
  (reduce-kv
    (fn [m k v]
      (assoc m k (prismatic-schema v)))
    {}
    schema))

(defn validator [schema]
  (ps/checker (prismatic-schema schema)))

(defn valid? [schema value]
  (if-let [error (ps/check (prismatic-schema schema) value)]
    (do (log/warn "Value does not match schema" schema value error) nil)
    true))
