(ns plinth.cache
  (:require
    [com.netflix.hystrix.core :as hystrix :refer [defcommand]]))

(defprotocol Cache
  "Key/value cache with optional expiration time"
  (-fetch [cache key])
  (-put [cache key value expires])
  (-delete [cache key]))

(extend-protocol Cache
  nil
  (-fetch [cache key] nil)
  (-put [cache key value expires] nil)
  (-delete [cache key] nil))

(defcommand fetch
  "Fetch value for key"
  {:hystrix/fallback-fn (constantly nil)}
  [cache key]
  (-fetch cache key))

(defcommand put
  "Put value into cache for key. Expires is the number of seconds before the value is evicted.  If nil or 0, the value may stay in the cache until evicted through normal memory pressure."
  {:hystrix/fallback-fn (constantly nil)}
  [cache key value & [expires]]
  (-put cache key value (or expires 0)))

(defcommand delete
  "Evict value for key. Waits until value is evicted."
  {:hystrix/fallback-fn (constantly nil)}
  [cache key]
  (-delete cache key))

(defmacro through
  "Fetch data from the cache if available, or load from storage otherwise.  For example:

    (through key 60
      (load-data key))

  will try to load from cache, otherwise execute the body `(load-data key)`, cache the result with an expiration of 60 seconds, and return the result."
  ([cache key load]
    `(through ~cache ~key 0 ~load))
  ([cache key expires load]
    `(if-let [rt# (fetch ~cache ~key)]
      rt#
      (let [v# ~load]
        (when v#
          (put ~cache ~key v# ~expires))
        v#))))
