(ns plinth.db
  (:require
    [clj-time.core :as t]
    [clj-time.coerce :as c]
    [plinth.metrics :as metrics]
    [clojure.java.jdbc :as jdbc]
    [clojure.tools.logging :as log]
    [com.stuartsierra.component :as component]
    [com.netflix.hystrix.core :as hystrix :refer [defcommand]])
  (:import
    [java.util Date Calendar TimeZone]
    [org.joda.time DateTime LocalTime]
    [java.sql BatchUpdateException PreparedStatement ResultSet SQLException SQLTransientConnectionException Statement]
    [com.jolbox.bonecp BoneCPDataSource]))

(def ^:private utc-calendar (Calendar/getInstance (TimeZone/getTimeZone "UTC")))

(extend-protocol clj-time.coerce/ICoerce
  org.joda.time.LocalTime
  (to-date-time [local-time]
    (t/plus (t/epoch) (t/millis (.getMillisOfDay local-time)))))

(extend-protocol clojure.java.jdbc/ISQLParameter
  java.util.Date
  (set-parameter [value statement idx]
    (.setTimestamp statement idx (c/to-timestamp value) utc-calendar))

  org.joda.time.DateTime
  (set-parameter [value statement idx]
    (.setTimestamp statement idx (c/to-timestamp value) utc-calendar))

  org.joda.time.LocalDateTime
  (set-parameter [value statement idx]
    (.setTimestamp statement idx (c/to-timestamp value) utc-calendar))

  org.joda.time.LocalTime
  (set-parameter [value statement idx]
    ;; clj-time.coerce/to-sql-time returns a java.sql.Timestamp. WTF?
    (.setTime statement idx (java.sql.Time. (.getTime (c/to-sql-time value))) utc-calendar))

  org.joda.time.LocalDate
  (set-parameter [value statement idx]
    (.setDate statement idx (c/to-sql-date value) utc-calendar))

  java.net.URI
  (set-parameter [value statement idx]
    (.setObject statement idx (.toString value)))

  java.net.URL
  (set-parameter [value statement idx]
    (.setObject statement idx (.toString value))))

(extend-protocol clojure.java.jdbc/IResultSetReadColumn
  java.sql.Array
  (result-set-read-column [object metadata idx]
    (into [] (.getArray object)))

  java.sql.Date
  (result-set-read-column [object metadata idx]
    (c/to-local-date (c/from-sql-date object)))

  java.sql.Time
  (result-set-read-column [object metadata idx]
    ;; there is no clj-time.coerce/to-local-time. WTF?
    (org.joda.time.LocalTime. (c/from-sql-time object)))

  java.sql.Timestamp
  (result-set-read-column [object metadata idx]
    (c/from-date object)))

(defn- make-prepared-statement [connection sql params]
  (let [statement (.prepareStatement connection sql)]
    (dorun
      (map-indexed
        (fn [idx param]
          (jdbc/set-parameter param statement (inc idx)))
        params))
    statement))

(defn- execute-fragment [{:keys [metrics] :as db} connection identifier-fn fragment]
  (log/trace "Executing SQL" fragment)
  (metrics/time! metrics ["plinth.db" "sql-statement" (:sql fragment)]
    (let [statement (make-prepared-statement connection (:sql fragment) (:parameters fragment))]
      (if (.execute statement)
        (vec (jdbc/result-set-seq (.getResultSet statement) :identifiers identifier-fn))
        (.getUpdateCount statement)))))

(defcommand transact!
  "Run the SQLFragments in a transaction"
  [{:keys [datasource] :as db} identifier-fn & fragments]
  (if-let [connection (.getConnection datasource)]
    (try
      (.setAutoCommit connection false)
      (mapv #(execute-fragment db connection identifier-fn %) (remove nil? fragments))
      (catch SQLException e
        (log/warn e "Exception while executing SQL")
        (try
          (log/info "Rolling back commit")
          (.rollback connection)
          (catch SQLException e2
            (log/warn e2 "Exception while rolling back commit"))))
      (finally
        (.setAutoCommit connection true)
        (.close connection)))
    (throw (SQLTransientConnectionException. (str "Unable to get SQL connection to database: " (pr-str db))))))

(defcommand query
  "Query the database with a SQLFragment"
  { :hystrix/fallback-fn (constantly nil) }
  [db identifier-fn fragment]
  (first (transact! db identifier-fn fragment)))

(defn- contains-string? [substring string]
  (<= 0 (.indexOf string substring)))

(defn- driver-class-for-uri [uri]
  (condp contains-string? uri
    "postgresql"  "org.postgresql.Driver"
    ""))

(defrecord SQLDatabase [host port db user password]
  component/Lifecycle
  (start [this]
    (log/trace "Starting up connection to SQL database" host ":" port "as" user)
    (Class/forName "org.postgresql.Driver")
    (let [datasource
          (doto
            (BoneCPDataSource.)
            (.setDriverClass "org.postgresql.Driver")
            (.setJdbcUrl (str "jdbc:postgresql://" host ":" port "/" db))
            (.setUsername user)
            (.setPassword password))]
    (assoc this :datasource datasource)))
  (stop [this]
    (when (:datasource this)
      (.close (:datasource this)))
    (dissoc this :datasource)))
