(ns plinth.schema.coerce
  (:require
    [clj-time.coerce :as c]
    [plinth.schema :as schema])
  (:import
    [java.net URI]
    [java.util UUID Date]))

(defn- to-integer [number]
  (cond
    (nil? number)    nil
    (number? number) (int number)
    (string? number) (Integer/parseInt number)
    :else            (throw (NumberFormatException.))))

(defn- to-double [number]
  (try
    (cond
      (nil? number)    nil
      (number? number) (double number)
      (string? number) (Double/parseDouble number)
      :else            Double/NaN)
    (catch Exception e
      Double/NaN)))

(defn- to-uuid [uuid]
  (cond
    (nil? uuid)           nil
    (instance? UUID uuid) uuid
    (string? uuid)        (UUID/fromString uuid)
    :else                 (throw (ex-info "Could not convert to UUID" {:value uuid}))))

(defn- to-date [date]
  (cond
    (nil? date)           nil
    (instance? Date date) date
    (integer? date)       (Date. date)
    (string? date)        (c/to-date (c/from-string date))
    :else                 (throw (ex-info "Could not convert to Date" {:value date}))))

(defn- to-uri [uri]
  (cond
    (nil? uri)          nil
    (string? uri)       (URI. uri)
    (instance? URI uri) uri
    :else               (throw (ex-info "Could not convert to URI" {:value uri}))))

(defmulti coerce #'schema/dispatch)
(defmethod coerce :any [schema from] from)
(defmethod coerce :serial [schema from] (to-integer from))
(defmethod coerce :string [schema from] (str from))
(defmethod coerce :number [schema from] (to-double from))
(defmethod coerce :integer [schema from] (to-integer from))
(defmethod coerce :uuid [schema from] (to-uuid from))
(defmethod coerce :instant [schema from] (to-date from))
; (defmethod coerce :time [schema] (wrap-coerce schema ps/Str))
; (defmethod coerce :date [schema] (wrap-coerce schema ps/Str))
(defmethod coerce :boolean [schema from] (boolean from))
(defmethod coerce :uri [schema from] (to-uri from))
(defmethod coerce :constant [schema from]
  (when (not= from (:subschema schema))
    (throw (ex-info "Invalid constant" {:value from})))
  from)
; (defmethod coerce :enum [schema] (wrap-coerce schema (apply ps/enum (:subschema schema))))
; (defmethod coerce :set [schema]
;   (wrap-coerce schema (set (map coerce (:subschema schema)))))
; (defmethod coerce :sequence [schema]
;   (wrap-coerce schema
;     (mapv #(ps/one (coerce %) (name (:name % (:type % :unknown)))) (:subschema schema))))
; (defmethod coerce :value [schema]
;   (wrap-coerce schema (coerce (:subschema schema))))
(defmethod coerce :clj-map [schema from]
  (reduce-kv
    (fn [m k v]
      (assoc m k (coerce v (get from k))))
    {}
    schema))
