(ns plinth.login.password
  (:require
    [plinth.schema :as s]
    [plinth.table :as table]
    [clojure.tools.logging :as log]
    [clojurewerkz.scrypt.core :as sc]
    [com.stuartsierra.component :as component]))

(def PasswordLoginSchema
  (s/table :password-logins :username
    { :username (assoc s/Str :required true :unique true)
      :encrypted-password (s/required s/Str)
      :user-id s/Str }))

(def invalid-username   "Invalid username")
(def duplicate-username "Username already taken")
(def invalid-password   "Invalid username or password")

(defn- encrypt [password]
  (sc/encrypt password 16384 8 1))

(defn username-exists? [table username]
  (boolean (table/look-up table username)))

(defn find-user-id-by-username [table username]
  (:user-id (table/look-up table username)))

(defn create! [table user-id username password]
  (log/debug "Creating login for" username)
  (when-not
    (table/insert! table
      { :user-id user-id
        :username username
        :encrypted-password (encrypt password) })
    (throw (ex-info duplicate-username {:username username}))))

(defn login [table username password]
  (log/debug "Logging in" username)
  (let [{:keys [user-id encrypted-password]} (table/look-up table username)]
    (if (sc/verify password encrypted-password)
      user-id
      (throw (ex-info invalid-password {:username username})))))

(defn reset-password! [table username password]
  (log/debug "Resetting password for" username)
  (when-not (table/update! table username {:encrypted-password (encrypt password)})
    (throw (ex-info invalid-username {:username username}))))

(defn update-username! [table old-username new-username]
  (log/debug "Updating username from" old-username "to" new-username)
  (when-not (table/update! table old-username {:username new-username})
    (throw (ex-info invalid-username {:old-username old-username :new-username new-username}))))

(defn delete! [table username]
  (log/debug "Deleting login for username" username)
  (when-not (table/delete! table username)
    (throw (ex-info invalid-username {:username username}))))
