(ns plinth.schema
  (:import
    [org.apache.commons.validator.routines EmailValidator CreditCardValidator]))

(defn valid-email? [email]
  (.isValid (EmailValidator/getInstance) email))

(let [cc-validator
      (CreditCardValidator.
        (+
          CreditCardValidator/AMEX
          CreditCardValidator/DINERS
          CreditCardValidator/DISCOVER
          CreditCardValidator/MASTERCARD
          CreditCardValidator/VISA))]
  (defn valid-credit-card? [cc]
    (.isValid cc-validator cc)))

(def types #{
  :table
  :relation
  :any
  :enum
  :bytes
  :serial
  :string
  :boolean
  :integer
  :number
  :instant
  :time
  :date
  :uuid
  :uri
  :value
  :constant
  :sequence
  :set
  ; :variant
  ; :decimal
  })

(defrecord Schema [type subschema predicate required cardinality ordered unique name belongs-to?])

(def Id         (map->Schema {:type :serial :required true :unique true}))
(def Any        (map->Schema {:type :any}))
(def Str        (map->Schema {:type :string}))
(def Uuid       (map->Schema {:type :uuid}))
(def Int        (map->Schema {:type :integer}))
(def Num        (map->Schema {:type :number}))
(def Bool       (map->Schema {:type :boolean}))
(def Timestamp  (map->Schema {:type :instant}))
(def Date       (map->Schema {:type :date}))
(def Time       (map->Schema {:type :time}))
(def Url        (map->Schema {:type :uri}))
(def Email      (map->Schema {:type :string :predicate valid-email?}))
(def CreditCard (map->Schema {:type :string :predicate valid-credit-card?}))

(defn table [table-name primary-key schema]
  (map->Schema {:type :table :subschema schema :name table-name :primary-key primary-key}))

(defn belongs-to [relation]
  (map->Schema {:type :relation :to relation :belongs-to? true}))
(defn related-to [relation]
  (map->Schema {:type :relation :to relation}))

(defn required [schema] (assoc schema :required true))
(defn sequence-of [schema] (map->Schema {:type :sequence :subschema schema}))

(defn- dispatch [schema & _]
  (cond
    (instance? Schema schema)
      (:type schema :default)
    (map? schema)
      :clj-map
    :else
      :default))
