(ns healthcheck.core
  "Health checks for Ring, following the reloaded pattern.
   Provides a component for configuration and a configurable endpoint.
  "
  (:require
    [com.stuartsierra.component :as component]
    [compojure.api.sweet :refer [GET routes]]
    [ring.util.http-response :refer :all]))

(defrecord Healthcheck [config]
  component/Lifecycle
  (start [this] this)
  (stop [this]this))

(defn healthcheck [config]
  (->Healthcheck config))


(defn default-healthchecks []
  {::basic (fn [] "OK")})

(defn healthcheck-result [check]
  (try
    (let [check-result (check)]
      (if check-result
        [::success check-result]
        [::failure check-result]))
    (catch Exception e
      [::failure e])))

(defn success?
  "Returns true if ALL checks succeed."
  [healthcheck-results]
  (empty? (filter #(not= ::success (first (second %))) healthcheck-results)))

(defn run [healthcheck-map]
  (into {} (map (fn [[name check]] [name (healthcheck-result check)])
                healthcheck-map)))

(defn endpoint
  "Creates a healthcheck route for Ring. Configuration may be passed to config
  in the :healthcheck key.

  Configuration options:

    - healthchecks:     A map of keyword check name and check-fn pairs. A var
                        may also be passed for usage with Duct or similar
                        system declarations.
    - path              The path of the healthcheck route. Default \"/_health\"

  Example:
      (endpoint {:healthcheck
                 {:config {:healthchecks default-healthchecks
                           :path         \"/_health\"}}})"
  [{{{healthchecks :healthchecks
      path :path
      :or  {healthchecks default-healthchecks
            path "/_health"}}
     :config}
    :healthcheck}]
  (let [checks (healthchecks)]
    (routes
      (GET path []
           :name    :healthcheck
           :summary "Returns healthcheck information."
           (let [result (run checks)]
             (if (success? result)
               (ok result)
               (internal-server-error! result)))))))
