(ns singularity.service

  "Communicates with the Singularity REST API."

  (:use
    singularity.request)
  (:require
    [clojure.string :as str]
    [clojure.data.json :as json]
    [net.cgrand.enlive-html :as enlive])
  (:import
    brilliantarc.singularity.SingularityException
    [java.io StringReader]))

(defmacro with-credentials
  [credentials & body]
  `(binding [singularity.request/*credentials* ~credentials]
     (do ~@body)))

(defn p
  "Return the portfolio slug for the meme, or nil if it's not present."
  [meme]
  (let [{{slug :slug} :portfolio} meme]
    slug))

(defn d
  "Return the definition name for the meme, or nil if it's not present."
  [meme]
  (let [{{slug :name} :definition} meme]
    slug))

(defn authenticate
  "Authenticate a user account and return the user's credentials for future API
  calls."
  [login password]
  (POST (to "user_session") {:login login :password password}))

(defn authentic
  "Check the user's credentials with the server, to ensure that they are valid.
  Must be run from within a with-credentials body."
  []
  (GET (to "user_session")))

(defn uuid
  "Call the server to generate a UUID.  Can be helpful in some environments to
  have the IDs generated on the same server."
  []
  (:uuid (GET (to "uuid"))))

(defn slug
  "Generate an SEO-compatible slug using the given string."
  [to-slugify]
  (:slug (GET (to "slugify") {:value to-slugify})))

(defn reset-test
  "Reset the test porfolio, i.e. wipe it."
  []
  (GET (to "test" "reset_terra")))

(defn definitions
  []
  (GET (to "d")))

(defn definition
  "Check if a definition exists."
  [name]
  (GET (to "d" name)))

(defn create-definition
  "Create a new definition.

    name:       a globally unique name for the definition
  "
  [name]
  (POST (to "d") {:name name}))

(defn portfolios
  "Return the list of existing porfolios in Singularity."
  []
  (GET (to "p")))

(defn portfolio
  "Look up the given portfolio's slug and return it."
  [slug]
  (GET (to "p" slug)))

(defn create-portfolio
  "Create a new portfolio.

    name:       a globally unique name for the portfolio
    slug:       an optional unique slug for the portfolio; generated from the
                name if not supplied
    clean:      if true, will format the name of the portfolio like a title,
                e.g. \"A SAMPLE PORTFOLIO\" becomes \"A Sample Portfolio\"
  "
  [name & [slug clean]]
  (POST (to "p") {:name name :slug slug :clean clean}))

(defn collections
  "Get the collections for the given portfolio."
  [portfolio]
  (GET (to "p" portfolio "c")))

(defn collection
  "Look up a collection by its slug.  Useful to see if the collection exists
  before trying to create it."
  [portfolio slug]
  (GET (to "p" portfolio "c" slug)))

(defn create-collection
  "Create a new collection.

    name:       the human-readable name of the collection
    slug:       an SEO-compatible identifier, unique within the context of a
                portfolio
    portfolio   the slug of the portfolio in which to add this collection
    clean:      if true, will format the name of the collection like a title,
                e.g. \"A SAMPLE COLLECTION\" becomes \"A Sample Collection\"
  "
  [{:keys [name slug portfolio]}]
  (POST (to "p" portfolio "c") {:name name :slug slug}))

(defn meme
  "Return the details about a meme identified by the portfolio, definition and
  slug."
  [portfolio definition slug]
  (GET (to "p" portfolio "d" definition "m" slug)))

(defn meme-by-external
  "Return the details about a meme identified by the portfolio, definition, and
  slug.  Presumes the external identifier is unique in the context of the
  portfolio and definition.  If this is not the case, the results of this call
  will be unpredictable."
  [portfolio definition external]
  (GET (to "p" portfolio "d" definition "m") {:external external}))

(defn meme-by-name
  "Look up the details of a meme by its name.  Returns the first match.  Note
  that memes are not guaranteed to have unique names, so unless your application
  is enforcing this manually, this function may return random results."
  [portfolio definition name]
  (GET (to "p" portfolio "d" definition "m" name) {:by "name"}))

(defn memes
  "Get the memes in a collection, portfolio, or of a certain definition in a
  portfolio.  Expects an indicator--:collection, :portfolio, :definition--and
  one or two objects.

    (memes :collection \"taxonomies\" :portfolio \"PKT\")
    (memes :portfolio \"PKT\")
    (memes :definition \"Category\" :portfolio \"PKT\")

  Note:  For :definition, the third parameter is the portfolio.  You may NOT
  look up all the memes for a single definition.
  "
  [& options]
  (let [options (apply hash-map options)
        {collection :collection portfolio :portfolio definition :definition} options]
    (cond
      (and collection portfolio) (GET (to "p" portfolio "c" collection "m"))
      (and definition portfolio) (GET (to "p" portfolio "d" definition "m"))
      portfolio (GET (to "p" portfolio "m")))))

(defn create-meme
  "Create a new meme.

    name:       the human-readable name of the meme
    slug:       an SEO-compatible identifier, unique within the context of a
                definition and portfolio
    external:   a third-party identifier for the meme; not required to be
                unique, but recommended (optional)
    definition: the name of a definition for the meme, e.g. \"Category\" or
                \"Tag\"
    language:   the ISO code for the language of the name of the meme (and
                possibly the slug), e.g. \"en\" or \"de\"; defaults to the
                language of the portfolio if nil or blank
    portfolio:  the slug of the portfolio in which to add this meme
    collection: the slug of the collection in which to add this meme (optional);
                must be in the same portfolio as the meme
    related-to: a map with a key indicating the relation or verb, and the value
                pointing to the meme to relate this meme to; this meme will be
                the subject, the related-to meme the object of the relation;
                you may also indicate :weight in this map, but no other
                properties
    clean:      if true, will format the name of the meme like a title,
                e.g. \"A SAMPLE MEME\" becomes \"A Sample Meme\"

  Note that you may EITHER indicate a :collection or a :related-to meme; you
  may not indicate both.  If you do, a SingularityException will be thrown.
  "
  [{:keys [name slug external definition language portfolio collection related-to clean]}]
  (if (and collection related-to)
    (throw (SingularityException. (str "Please indicate either a collection or another meme when creating " name " (" slug "), not both.") 500)))

  ;; Create the new meme in a collection?
  (if collection
    (POST (to "p" portfolio "c" collection "d" definition "m") {:name name :slug slug :external external :iso_639_1 language :clean clean})

    ;; Create the new meme and immediately relate it to something, like a parent category
    (if related-to
      (let [{weight :weight direction :direction :or (weight nil direction :outgoing)} related-to
            [verb object] (first (dissoc related-to :weight :direction))]
        (POST (to
                "p" (p object) "d" (d object) "m" (:slug object)
                "r" verb
                "p" portfolio "d" definition "m")
          {:name name :slug slug :external external :iso_639_1 language :inverse (= direction :incoming) :weight weight :clean clean}))

      ;; Or, just create a lonely ol' meme
      (POST (to "p" portfolio "d" definition "m") {:name name :slug slug :external external :iso_639_1 language :clean clean}))))

(defn update-meme
  "Update a meme in Singularity.  You may change a meme's name, language,
  external identifier, and any custom properties.  You may not change its
  portfolio, slug, or definition.

  TODO:  Custom properties have not been implemented yet."
  [portfolio definition slug version name & [language external clean]]
  (PUT (to "p" portfolio "d" definition "m" slug) {:name name :iso_639_1 language :external external :version version :clean clean}))

(defn delete-meme
  "Delete a meme completely from Singularity.  All relations to this meme will
  be lost.  Requires the version of the meme as well, to ensure you don't try
  to delete an old version of a meme inappropriately."
  [portfolio definition slug version]
  (DELETE (to "p" portfolio "d" definition "m" slug) {:version version}))

(defn relate-memes
  "Relate two memes together using the given relation (verb)."
  [subject verb object & [properties]]
  (PUT (to
         "p" (p subject) "d" (d subject) "m" (:slug subject)
         "r" (name verb)
         "p" (p object) "d" (d object) "m" (:slug object))
    properties))

(defn relate-many
  "Relate a number of memes to another.  Create \"sentences\" for each relation,
  a triple of [subject, verb, object] (and optionally, weight).  Verb is
  expected to be a string or keyword."
  [sentences]
  (POST (to "bulk") {:task "create" :data (json/json-str sentences)}))

(defn unrelate-memes
  "Break the relation between two memes using the given relation (verb)."
  [subject verb object]
  (DELETE (to
            "p" (p subject) "d" (d subject) "m" (:slug subject)
            "r" (name verb)
            "p" (p object) "d" (d object) "m" (:slug object))))

(defn unrelate-many
  "Break the relations between a number of memes.  Create \"sentences\" for each
  relation to be destroyed, as a triple of [subject, verb, object].  Verb is
  expected to be a string or keyword."
  [sentences]
  (POST (to "bulk") {:task "delete" :data (json/json-str sentences)}))

(defn unrelate-all
  "Break all the relations of the given type to or from this meme, based on
  direction."
  [meme verb & [direction]]
  (DELETE (to "p" (p meme) "d" (d meme) "m" (:slug meme) "r" (name verb)) {:inverse (= direction :incoming)}))

(defn related-memes
  "Get the memes related to the given meme by the given relation/verb."
  [meme verb & [direction]]
  (GET (to "p" (p meme) "d" (d meme) "m" (:slug meme) "r" verb) {:inverse (= direction :incoming)}))

(defn relations
  "Get the relations associated with the given meme."
  [meme & [direction]]
  (GET (to "p" (p meme) "d" (d meme) "m" (:slug meme) "r") {:inverse (= direction :incoming)}))

(defn collect-meme
  "Add a meme to a collection.  You may only add a meme to a collection if the
  meme and collection are in the same portfolio."
  [collection meme]
  (PUT (to "p" (p collection) "c" (:slug collection) "d" (d meme) "m" (:slug meme))))

(defn uncollect-meme
  "Remove a meme from a collection.  You may only remove memes from collections
  that are in the same portfolio."
  [collection meme]
  (DELETE (to "p" (p collection) "c" (:slug collection) "d" (d meme) "m" (:slug meme))))

(defn bulk-upload
  "Create the given memes, and relation them to the given category by their properties.
  Expects a list of memes, with relations attached for each (relation -> object):

    [{
      :name \"Some Option\",
      :slug \"optional-slug\",
      :definition \"Option\",
      :portfolio \"brilliant-arc\",
      :iso_639_1 \"en\",
      :external \"some-external-id\",
      :properties { :abc \"a custom property\" },
      :relations [
        { \"child-of\": { :slug \"restaurants\", :definition \"Category\", :portfolio \"brilliant-arc\" } },
        { \"mapped-to\": { :slug \"food-service\", :definition \"NAICS Category\", :portfolio \"NAICS\" } }
      ],
      :collections [
        { :slug \"collection-a\", :portfolio \"brilliant-arc\" }
      ]
    }]

  Relations and collections are optional.  If not provided, the meme is just created
  in the portfolio.

  The objects of any relations must already exist before this bulk operation is
  attempted.  You may not relate memes to other memes in this bulk upload.  If
  you need to do that, first bulk upload the memes without relations, then create
  the necessary relations after the memes have been successfully created.

  Either everything is saved to the graph, or nothing is saved and error messages are
  returned with each meme.  For example, if there already exists \"optional-slug\", the
  user will receive a 406, Not Acceptable, error, with something like:

    [{
      :name \"Some Option\",
      :slug \"optional-slug\",
      :definition \"Option\",
      :portfolio \"brilliant-arc\",
      :iso_639_1 \"en\",
      :external \"some-external-id\",
      :properties { :abc \"a custom property\" },
      :relations [
        { \"child-of\": { :slug \"restaurants\", :definition \"Category\", :portfolio \"brilliant-arc\" } },
        { \"mapped-to\": { :slug \"food-service\", :definition \"NAICS Category\", :portfolio \"NAICS\", :weight 0.4 } }
      ],
      :collections [
        { :slug \"collection-a\", :portfolio \"brilliant-arc\" }
      ],
      :error {
        :message \"There is already an Option in Brilliant Arc with the slug 'optional-slug'.\",
        :duplicate { ... the duplicate meme ... }
      }
    }]

  Returns the memes that were created if successful, or the JSON marked up with error
  messages if problems occurred."
  [memes]
  (POST (to "bulk") {:task "upload" :data (json/json-str memes)}))

(defn traversal
  "Call the traversal function without a starting meme.  Typically this means
  the starting meme is static within the function itself."
  [traversal-name & [parameters]]
  (PUT (to "t" traversal-name) parameters))

(defn traversal-from
  "Call the traversal function with the starting meme."
  [traversal-name meme & [parameters]]
  (PUT (to "p" (p meme) "d" (d meme) "m" (:slug meme) "t" traversal-name) parameters))

(defn stories
  "Get the stories available in the given portfolio."
  [portfolio & [as-nodes]]
  (GET (to "p" portfolio "stories") {:nodes as-nodes}))

(defn create-story
  "Create a new story in the given portfolio.  The slug is optional (and must
  be unique to the portfolio) but the name is not.

    name:       the human-readable name of the story
    slug:       an SEO-compatible identifier, unique within the context of the
                portfolio's stories
    portfolio:  the slug of the portfolio in which to add this story
    nodes:      return the representation of this story as a node, rather than
                a story object (internal use only)
    existing:   the slug of another story in the same portfolio; including this
                will copy the blocks from that story to this one
    clean:      will automatically clean up and titleize the name on the server
  "
  [{:keys [name slug portfolio nodes existing clean]}]
  (POST (to "p" portfolio "stories") {:name name :slug slug :nodes nodes :existing existing :clean clean}))

(defn update-story
  "Update the name of a story."
  [story name & [as-nodes clean]]
  (PUT (to "p" (p story) "stories" (:slug story)) {:name name :nodes as-nodes :clean clean}))

(defn delete-story
  "Delete a story completely from Singularity."
  [story]
  (DELETE (to "p" (p story) "stories" (:slug story))))

(defn available-blocks
  "Return all the available blocks in Singularity."
  []
  (GET (to "blocks")))

(defn blocks
  "Get the blocks for the given story, in order of priority."
  [story]
  (GET (to "p" (p story) "stories" (:slug story) "blocks")))

(defn create-block
  "Create a new block.  The slug is not required, but both the name and slug
  must be unique throughout the entire graph.

    name:       the human-readable name of the block
    slug:       an SEO-compatible identifier, unique across the entire graph
    existing:   the slug of another block; including this will copy the fields
                from that block to this one
    clean:      will automatically clean up and titleize the name on the server
  "
  [{:keys [name slug existing clean]}]
  (POST (to "blocks") {:name name :slug slug :existing existing :clean clean}))

(defn update-block
  "Update not only the name of a block, but is fields as well, in a single
  transaction."
  [updated]
  (let [{slug :slug name :name} updated
        fields (map (fn [field]
                      (let [{{language :iso_639_1} :language} updated]
                        (assoc field :language language))) (:fields updated))]

    (PUT (to "blocks" (:slug updated)) {:name name :slug slug :fields (json/json-str fields)})))

(defn add-block-to-story
  "Associate a block with a story.  If the priority is not provided or less
  than zero, or the priority is greater than the number of existing blocks in
  the story, the block is added to the end of the list.

  Note that a block may only exist once in a story.  If you add a block to a
  story a second time, it merely reprioritizes the block."
  [story block & [priority]]
  (PUT (to "p" (p story) "stories" (:slug story) "blocks" (:slug block)) {:priority priority}))

(defn remove-block-from-story
  "Remove the given block from the story."
  [story block]
  (DELETE (to "p" (p story) "stories" (:slug story) "blocks" (:slug block))))

(defn delete-block
  "Completely delete a block.  Of course this removes it from all stories as
  well."
  [block]
  (DELETE (to "blocks" (:slug block))))

(defn block-usage
  "Return the stories using this block."
  [block & [as-nodes]]
  (GET (to "blocks" (:slug block) "stories") {:nodes as-nodes}))

(defn available-fields
  "Return all the fields available in Singularity."
  []
  (GET (to "fields")))

(defn fields
  "Get the fields for the given block, in order of priority."
  [block]
  (GET (to "blocks" (:slug block) "fields")))

(defn create-field
  "Create a new field."
  [{:keys [name slug value_type minimum_value_length maximum_value_length valid_regex language required flags portfolio relation clean]}]
  (POST (to "fields") {:name name
                       :slug slug
                       :value_type value_type
                       :minimum_value_length minimum_value_length
                       :maximum_value_length maximum_value_length
                       :valid_regex valid_regex
                       :language language
                       :required required
                       :flags (json/json-str flags)
                       :portfolio portfolio
                       :relation relation
                       :clean clean}))

(defn update-field
  "Update an existing field."
  [slug & [{:keys [name value_type minimum_value_length maximum_value_length valid_regex language required flags portfolio relation clean]}]]
  (PUT (to "fields" slug) {:name name
                           :value_type value_type
                           :minimum_value_length minimum_value_length
                           :maximum_value_length maximum_value_length
                           :valid_regex valid_regex
                           :language language
                           :required required
                           :flags (json/json-str flags)
                           :portfolio portfolio
                           :relation relation
                           :clean clean}))

(defn add-field-to-block
  "Associate a field with a block.  Note that a field may only appear once in
  a block.  If you add a field a second time, you are merely adjusting its
  priority."
  [block field & [priority]]
  (let [{{language :iso_639_1} :language} field]
    (PUT (to "blocks" (:slug block) "fields" (:slug field)) {:priority priority :language language})))

(defn remove-field-from-block
  "Remove the given field from the block."
  [block field]
  (DELETE (to "blocks" (:slug block) "fields" (:slug field))))

(defn delete-field
  "Completely delete a field, which of course removes it from all blocks as
  well."
  [field]
  (DELETE (to "fields" (:slug field))))

(defn stories-xml
  "Get the XML for the stories associated with the given meme.  The result will
  be an XML document in a string, for parsing as you see fit."
  [meme]
  (GET (to "p" (p meme) "d" (d meme) "m" (:slug meme) "t" "edsa::inherited_stories.xml") {} :string))

(defn release-story
  "Release a version of a story XML document with the server, for historic and
  change tracking purposes.  Becomes part of the \"biography\" of the stories.

    meme:   the meme the XML document is related to
    xml:    the XML document to record; expects a \"created\" attribute on the
            root
    name:   a brief name for this release or version
    clean:  clean up the name of the biography
  "
  [meme xml name & [clean]]
  (let [{{created :created} :attrs} (enlive/xml-resource (StringReader. xml))]
    (POST (to "p" (p meme) "d" (d meme) "m" (:slug meme) "b") {:xml xml :name name :created created :clean clean})))

(defn biographies
  "Look up the story releases for a meme."
  [meme]
  (GET (to "p" (p meme) "d" (d meme) "m" (:slug meme) "b")))

(defn portfolio-biographies
  "Return the history of story releases for all the memes in the given
  portfolio.  If you would like to include the XML documents associated with
  the each biography record, pass xml in as true."
  [portfolio & [xml]]
  (GET (to "p" (:slug portfolio) "b") {:xml (true? xml) :latest true}))

(defn biography
  "Get the details about a single story release (biography).  The created
  parameter should be a UNIX timestamp."
  [meme created]
  (GET (to "p" (p meme) "d" (d meme) "m" (:slug meme) "b" created)))

(defn user-history
  "Get the history for the given user.

    login:    the user's account login
    from:     the starting point for results; defaults to 0
    max:      the maximum number of results to return; defaults to 100
  "
  [login & [from max]]
  (GET (to "s" login "history") {:offset from :limit max}))

(defn portfolio-history
  "Get the history for the given portfolio.

    portfolio:  the portfolio to find (slug)
    from:       the starting point for results; defaults to 0
    max:        the maximum number of results to return; defaults to 100
  "
  [portfolio & [from max]]
  (GET (to "p" portfolio "history") {:offset from :limit max}))

(defn users
  "Get a list of all the users in the system.  Will not return any secure
  information."
  []
  (GET (to "users")))

(defn user
  "Look up a user by his or her login."
  [login]
  (GET (to "users" login)))

(defn create-user
  "Create a new user account in Singularity.

  If you include a portfolio, the user will be attached to that portfolio.
  This typically limits activity of the user to the portfolio, based on the
  permissions configuration for the Singularity installation.

  You must have permission on the server to create users."
  [login email password password-confirmation & [portfolio skip-notification]]
  (let [options {:login login
                 :email email
                 :password password
                 :password_confirmation password-confirmation
                 :skip_notification skip-notification
                 :title "[TERRA] Brilliant Arc Terra Taxonomy Editor Account Created"}]
    (if portfolio
      (POST (to "p" portfolio "users") options)
      (POST (to "users") options))))

(defn update-email
  "Update the user's email address.  You must have permission on the server to
  update this user's account."
  [login email & [skip-notification]]
  (PUT (to "users" login) {:email email}))

(defn update-password
  "Update the user's password.  Must be the user, or have permission to update
  the user (as an admin, for example)."
  [login original-password password password-confirmation & [skip-notification]]
  (PUT (to "users" login) {:new_password password
                           :new_password_confirmation password-confirmation
                           :original_password original-password
                           :skip_notification skip-notification}))

(defn update-user
  "Update another user account, typically as an administrator.  If any value is
  nil, it will be ignored.  You must have permission on the server to perform
  this action."
  [login email & [password password-confirmation skip-notification]]
  (PUT (to "users" login) {:email email
                           :password password
                           :password_confirmation password-confirmation
                           :skip_notification skip-notification
                           :title "[TERRA] Brilliant Arc Terra Taxonomy Editor Account Updated"}))

(defn add-user-to-portfolio
  "Add the user to the given portfolio."
  [portfolio login]
  (PUT (to "p" portfolio "users" login)))

(defn remove-user-from-portfolio
  "Remove the user from the portfolio."
  [portfolio login]
  (DELETE (to "p" portfolio "users" login)))

(defn roles
  "Get the available roles."
  []
  (GET (to "d" "Role" "m")))

(defn add-user-to-role
  "Add the user to the given role."
  [role login]
  (PUT (to "users" login "roles" role)))

(defn remove-user-from-role
  "Remove the user from the given role."
  [role login]
  (DELETE (to "users" login "roles" role)))

(defn enable-user
  "Enable a user's account, i.e. allow the user to login and use the Terra
  tools."
  [login & [skip-notification]]
  (PUT (to "users" login) {:enable true
                           :skip_notification skip-notification
                           :title "[TERRA] Brilliant Arc Terra Taxonomy Editor Account Enabled"}))

(defn disable-user
  "Disable a user's account and prevent them from using the Terra tools."
  [login & [skip-notification]]
  (DELETE (to "users" login) {:skip_notification skip-notification
                              :title "[TERRA] Brilliant Arc Terra Taxonomy Editor Account Disabled"}))

(defn send-message
  "Send a message to the given user from the current user account, via
  Singularity"
  [to subject message]
  (let [subject (or subject (str "[TERRA] Message Received From " (:login (authenticate))))]
    (POST (to "users" to "messages") {:message message :title subject})))

(defn reset-password
  "Send the user a link to reset his or her password (forgot password)."
  [email]
  (POST (to "forgot_password") {:email email}))

(defn search
  "Search for memes (categories, properties, options) in a portfolio.

    portfolio:    the portfolio in which to search
    language:     limit search to this language (required, due to search
                  semantics)
    definitions:  limit the search to memes of these types (expects list)
    query:        the search phrase
    from:         starting number of results to return, for pagination; defaults
                  to 0
    max:          the maximum number of results to return; defaults to 10

  Returns a map of three values, :facets, :hits, and :total.  The :hits value
  will be the array of node objects representing the search results.  The :total
  value will be the total number of results possibly returned, if you were to
  return all results at once.  The :facets value will be a map of definitions
  to the number of results returned for that type of definition."
  [language query & [{:keys [portfolio definitions from max]}]]
  (GET (to "q") {:s query
                 :portfolio portfolio
                 :language language
                 :definitions (if (string? definitions) definitions (str/join "," definitions))
                 :from (or from 0)
                 :size (or max 10)}))
