(ns ^{:doc "Utility functions to deal with partitions"
      :author "Bruno Kim Medeiros Cesar"}
  loom.partitions)

; First, we need some definitions of data structures pertaining partitions.
; A partition is a way to separate elements in different groups, without letting anyone out.
;
; For example, a possible partition of the set of elements #{:a :b :c :d :e :f :g :h :i :j :k} is
;   [#{:a :b} #{:c :d :e :f} #{:g} #{:h} #{:i :j :k}]
;
;  We call this representation group vector. Although mathematically a partition is a set of groups,
; we store groups in a vector associate an index to each group, allowing to convert back and forth
; the other representations.
;  A more explicit representation may be given as a part-group map:
;   {0 #{:a :b}, 1 #{:c :d :e :f}, 2 #{:g}, 3 #{:h}, 4 #{:i :j :k}}
;  Our implementation treats this implementation and the previous as the same. This is called an
; index map.
;
; We can invert a group map to obtain an element-part map, which we call partition map:
;   {:a 0, :b 0, :c 1, :d 1, :e 1, :f 1, :g 2, :h 3, :i 4, :j 4, :k 4}
;
; Finally, we may convert a part-group map into a part-element multimap, stored simply as pairs:
;   #{[0 :a] [0 :b] [1 :c] [1 :d] [1 :e] [1 :f] [2 :g] [3 :h] [4 :i] [4 :j] [4 :k]}
; This representation is called partition multimap.
;
; Its important to sanity check each representation, to ensure that no element is repeated and that
; the partition indices increase monotonically from 0 to the maximum.

(declare check-index-map check-vector check-map check-multimap)
(declare index-map->vector vector->index-map
         vector->map map->vector
         map->multimap multimap->map
         vector->multimap multimap->vector)

;;;; Index map functions.
(defn check-index-map
  "Check solely that the indices in an index map range from 0 to the number of groups."
  [m]
  {:pre [(map? m)]}
  (= (range (count m))
     (sort (keys m))))

(defn index-map->vector
  "Converts a partition-group map to a group vector"
  [m]
  {:pre [(check-index-map m)]}
  (persistent!
   (reduce (fn [v [idx x]]
             (assoc! v idx x))
           (transient (vec (range (count m))))
           m)))

;;;; Vector functions.
(defn check-vector
  "Checks that a partition vector has no repeated elements and no empty groups."
  [part]
  {:pre [(or (vector? part) (map? part))]}
  (letfn [(check [v]
            (loop [acc #{}, xs v]
              (if (empty? xs)
                true
                (let [[x & xs] xs]
                  (if (or (empty? x) (some x acc))
                    false
                    (recur (into acc x) xs))))))]
    (if (vector? part)
      (check part)
      (and (check-index-map part)
           (check (index-map->vector part))))))

(defn vector->index-map
  "Converts a group vector to a partition-group map"
  [v]
  {:pre [(and (vector? v) (check-vector v))]}
  (into {} (map vector (range) v)))

(defn vector->map
  "Convers a group vector to an element-partition map"
  [v]
  {:pre [(check-vector v)]}
  )

(defn partition-map->vector
  "Converts a partition given as a {node part} map to a vector of sets.
  A partition identifier should be a number between in [0,p), where p
  is the number of partitions."
  [part]
  (let [p (inc (apply max -1 (vals part)))]
    (reduce (fn [set-vector [node part-index]]
              (update-in set-vector [part-index] conj node))
            (vec (repeat p #{}))
            part)))

(defn partition-vector->map
  "Converts a partition given as a vector of sets to a map of the form {node part}."
  [part]
  (reduce (fn [node-map [i vs]]
            (reduce #(assoc %1 %2 i) node-map vs))
          {}
          (map vector (range) part)))
