(ns bss.rampant.pagination
  (:refer-clojure :exclude [next])
  (:require [bss.rampant.utils :as u :refer [p p*]]
            [bss.rampant.sorting :refer [->lower compare-keys sort-by-keys]]))

;; Data windowing

(defn- paginate-range
  "Filter to the first elements between from and to, inclusive"
  [sorted-data sort-keys [from to] limit]
  (let [->tuple #(map (comp % ->lower) sort-keys)
        drop-fn (fn [item] (neg? ((compare-keys sort-keys) (->tuple item) from)))
        take-fn (fn [item] (>= 0 ((compare-keys sort-keys) (->tuple item) to)))]
    (->> sorted-data
         (remove drop-fn)
         (take-while take-fn)
         (take limit))))

(defn- paginate-after
  "Filter to the first elements after to, exclusive"
  [sorted-data sort-keys from limit]
  (let [->tuple #(map (comp % ->lower) sort-keys)
        drop-fn (fn [item] (>= 0 ((compare-keys sort-keys) (->tuple item) from)))]
    (->> sorted-data
         (remove drop-fn)
         (take limit))))

(defn- paginate-before
  "Filter to the last elements before from, exclusive"
  [sorted-data sort-keys to limit]
  (let [->tuple #(map (comp % ->lower) sort-keys)
        take-fn (fn [item] (neg? ((compare-keys sort-keys) (->tuple item) to)))]
    (->> sorted-data
         (take-while take-fn)
         (take-last limit))))

(defn- paginate-open
  "Filter to the first elements"
  [sorted-data limit]
  (take limit sorted-data))

(defn paginate [data sort-keys [from to :as cursor] limit]
  (when (seq data)
    (cond (or (nil? cursor) (and (not= '* from) (not= '* to)))
          (paginate-range data sort-keys cursor limit)

          (and (= '* from) (= '* to))
          (paginate-open data limit)

          (not= '* from)
          (paginate-after data sort-keys from limit)

          (not= '* to)
          (paginate-before data sort-keys to limit))))

(declare refine-cursor)

(def fast-sort (memoize sort-by-keys))

;; TODO: reorganise this to be less complected AND more efficient
(defn paginate-and-sort [data sort-keys cursor limit & [last-cursor]]
  (assert limit)
  (let [cursor (or cursor ['* '*])
        sorted (fast-sort sort-keys data)
        paged  (paginate sorted sort-keys cursor limit)
        cursor (refine-cursor sorted sort-keys cursor limit)]
    (if (and (empty? paged) last-cursor)
      (paginate-and-sort data sort-keys last-cursor limit)
      [cursor paged])))

;; Cursors

(defn refine-cursor
  "Convert open or one-sided cursors to range, given data"
  [data sort-keys [from to :as page-cursor] limit]
  (when (seq data)
    (doall
     (let [items (paginate (sort-by-keys sort-keys data) sort-keys page-cursor limit)]
       (if-not (seq items)
         (cond (and (= '* from) (not= '* to)) (refine-cursor data sort-keys ['* '*] limit)
               (and (not= '* from) (= '* to)) (refine-cursor data sort-keys [from from] limit)
               :else ['* '*])
         [(map (first items) (map ->lower sort-keys))
          (map (last items) (map ->lower sort-keys))])))))

(defn next-cursor [[_ to :as old]]
  (if (not= to '*)
    [to '*]
    old))

(defn prev-cursor [[from _ :as old]]
  (if (not= from '*)
    ['* from]
    old))


(comment

;; Better system:

;; types of data window:
;;
;; open (ie. ['* '*])
;; one-sided, exclusive (ie. [a '*] and ['* b])
;; range, inclusive (ie. [a b])
;; on-sided, inclusive (not currently modelled)

;; operations:
;;
;; next
;; prev
;; refine (this should convert to range AND expand to limit in certain cases)

;; open:
;; - next: fail
;; - prev: fail
;; - refine: take limit from start

;; one-sided:
;; - next: fail unless has to, in to case change from inclusive to exclusive
;; - prev: fail unless has from, in from case change from inclusive to exclusive
;; - refine: take limit towards '*

;; range:
;; - next: turn into exclusive one-sided
;; - prev: turn into exclusive one-sided
;; - refine: expand/shrink towards right


(declare data->pagination)

(declare ->PageAfter)

(defprotocol IPagination
  (next [_])
  (prev [_])
  (next-count [_ data])
  (prev-count [_ data])
  (refine [_ data]))

(defrecord Unrestricted []
  IPagination
  (next [this] (throw (ex-info "Not supported" this)))
  (prev [this] (throw (ex-info "Not supported" this)))
  (next-count [_ _] nil)
  (prev-count [_ _] nil)
  (refine [_ data]
    (data->pagination data)))

(defrecord PageBefore [value inclusive]
  IPagination
  (next [_] (->PageAfter value (not inclusive)))
  (prev [this] (throw (ex-info "Not supported" this)))
  (next-count [_ data] (count (remove (partial (if inclusive >= >) value) data)))
  (prev-count [_ _] nil)
  (refine [_ data]
    (data->pagination (remove (partial (if inclusive < <=) value) data))))

(defrecord PageAfter [value inclusive]
  IPagination
  (next [this] (throw (ex-info "Not supported" this)))
  (prev [_] (->PageBefore value (not inclusive)))
  (next-count [_ _] nil)
  (prev-count [_ data] (count (remove (partial (if inclusive <= <) value) data)))
  (refine [_ data]
    (data->pagination (remove (partial (if inclusive > >=) value) data))))

(defrecord Range [from to]
  IPagination
  (next [_] (->PageAfter to false))
  (prev [_] (->PageBefore from false))
  (next-count [_ data] (count (remove (partial >= to) data)))
  (prev-count [_ data] (count (remove (partial <= from) data)))
  (refine [_ data]
    (data->pagination
     ;; NB: not it does NOT limit on the right (greedy expansion)
     (->> data
          (remove (partial > from))))))

(defn data->pagination [data]
  (->Range (first data) (last (take 3 data))))

;; unrestricted can only refine, and takes until limit from start
(assert (= (->Range 0 2)
           (refine (->Unrestricted) (range))))

(-> (->Unrestricted)
    (refine [1 2 3 4])
    (next)
    (prev)
    (next)
    (refine [3 4 5 6 7 8])
    (next)
    (prev)
    (refine [3 4 5 6 7 8 9])
    (refine [2 4 7 9]))

(next-count (->Range 3 7) [1 2 3 4 5 6 7 8 9 10])
(prev-count (->Range 3 7) [1 2 3 4 5 6 7 8 9 10])

;; examples:

;; if client wants to keep all data on screen
;; - do not apply refine, data will stretch
;; - if wanting to only stretch to a point, can truncate or refine

;; can choose to refine on client or server - generally unless
;; explicitly in "offline" mode, want to also refine on server (to not miss
;; data)

;; without refining, we can safely go back/forward with cache without ever
;; missing data

;; can ping server (or be notified of changes) for next-count and prev-count
;; to drive breadcrumbs AND disabling of controls
)

;;;;;;;;;;;; This file autogenerated from src-cljx/bss/rampant/pagination.cljx
