(ns burningswell.api.airports
  "The airports in the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.db.airports :as airports]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn airport-not-found
  "Return a 404 response for an airport that could not be found by `id`."
  [id]
  (not-found (format "Airport %s not found" id)))

(defnk $GET
  "List all airports."
  {:responses {200 [Airport]}}
  [[:request query-params :- PaginationParams]
   [:resources api-client db]]
  (let [airports (airports/all db query-params)]
    (ok (hal/links api-client :airport airports))))

(defnk $POST
  "Create a new airport."
  {:responses {201 Airport}}
  [[:request body :- create-airport]
   [:resources api-client db broker]]
  (let [airport (airports/insert db body)
        airport (hal/link api-client :airport airport)]
    (publish broker "airports.created" airport)
    (created airport)))

(defnk $:id$GET
  "Show an airport."
  {:responses {200 Airport 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [airport (airports/by-id db id)]
    (ok (hal/link api-client :airport airport))
    (airport-not-found id)))

(defnk $:id$DELETE
  "Delete an airport."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [airport (airports/by-id db id)]
    (do (airports/delete db airport)
        (publish broker "airports.deleted" airport)
        (no-content))
    (airport-not-found id)))

(defnk $:id$PUT
  "Update an airport."
  {:responses {200 Airport 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-airport]
   [:resources api-client db broker]]
  (if-let [airport (airports/by-id db id)]
    (let [airport (airports/update db (merge airport body))
          airport (hal/link api-client :airport airport)]
      (publish broker "airports.updated" airport)
      (ok airport))
    (airport-not-found id)))
