(ns burningswell.api.comments
  "The comments of users on spots."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.comments :as comments]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(defn- comment-not-found
  "Return a 404 response for a comment that could not be found by `id`."
  [id]
  (not-found (format "Comment %s not found" id)))

(defnk $GET
  "List all comments."
  {:responses {200 [Comment]}}
  [[:request query-params :- CommentsParams]
   [:resources api-client db]]
  (let [comments (comments/all db query-params)
        comments (hal/links api-client :comment comments)]
    (ok comments) ))

(defnk $POST
  "Create a new comment."
  {:responses {201 Comment}}
  [[:request
    body :- create-comment
    identity :- User]
   [:resources api-client db broker]]
  (let [body (assoc-in body [:_embedded :user] identity)
        comment (comments/insert db body)
        comment (hal/link api-client :comment comment)]
    (publish broker "comments.created" comment)
    (created comment)))

(defnk $:id$GET
  "Show a comment."
  {:responses {200 Comment 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [comment (comments/by-id db id)]
    (ok (hal/link api-client :comment comment))
    (comment-not-found id)))

(defnk $:id$DELETE
  "Delete a comment."
  {:responses {204 s/Any 403 Forbidden 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [comment (comments/by-id db id)]
    (do (comments/delete db comment)
        (publish broker "comments.deleted" comment)
        (no-content))
    (comment-not-found id)))

(defnk $:id$PUT
  "Update a comment."
  {:responses {200 Comment 403 Forbidden 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-comment
    identity :- User]
   [:resources api-client db broker]]
  (if-let [comment (comments/by-id db id)]
    (let [body (-> (assoc body :id id)
                   (assoc-in [:_embedded :user] identity))
          comment (comments/update db body)
          comment (hal/link api-client :comment comment)]
      (publish broker "comments.updated" comment)
      (ok comment))
    (comment-not-found id)))
