(ns burningswell.api.continents
  "The continents of the world."
  (:require [burningswell.api.client :as api]
            [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.spots :refer [enhance-spots]]
            [burningswell.db.airports :as airports]
            [burningswell.db.continents :as continents]
            [burningswell.db.countries :as countries]
            [burningswell.db.ports :as ports]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.db.users :as users]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn- continent-not-found
  "Return a 404 response for a continent that could not be found by `id`."
  [id]
  (not-found (format "Continent %s not found" id)))

(defnk $GET
  "List all continents."
  {:responses {200 [Continent]}}
  [[:request query-params :- PaginationParams]
   [:resources api-client db]]
  (let [continents (continents/all db query-params)]
    (ok (hal/links api-client :continent continents))))

(defnk $POST
  "Create a new continent."
  {:responses {201 Continent}}
  [[:request body :- create-continent]
   [:resources api-client db broker]]
  (let [continent (continents/insert db body)]
    (publish broker "continents.created" continent)
    (created (hal/link api-client :continent continent))))

(defnk $:id$GET
  "Show a continent."
  {:responses {200 Continent 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [continent (continents/by-id db id)]
    (ok (hal/link api-client :continent continent))
    (continent-not-found id)))

(defnk $:id$DELETE
  "Delete a continent."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [continent (continents/by-id db id)]
    (do (continents/delete db continent)
        (publish broker "continents.deleted" continent)
        (no-content))
    (continent-not-found id)))

(defnk $:id$PUT
  "Update a continent."
  {:responses {200 Continent 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-continent]
   [:resources api-client db broker]]
  (if-let [continent (continents/by-id db id)]
    (let [continent (continents/update db (merge continent body))
          continent (hal/link api-client :continent continent)]
      (publish broker "continents.updated" continent)
      (ok continent))
    (continent-not-found id)))

(defnk $:id$airports$GET
  "List all airports on a continent."
  {:responses {200 [Airport] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [continent (continents/by-id db id)]
    (let [airports (airports/in-continent db continent query-params)]
      (ok (hal/links api-client :airport airports)))
    (continent-not-found id)))

(defnk $:id$countries$GET
  "List all countries on a continent."
  {:responses {200 [Country] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [continent (continents/by-id db id)]
    (let [countries (countries/in-continent db continent query-params)]
      (ok (hal/links api-client :country countries)))
    (continent-not-found id)))

(defnk $:id$ports$GET
  "List all ports on a continent."
  {:responses {200 [Port] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [continent (continents/by-id db id)]
    (let [ports (ports/in-continent db continent query-params)]
      (ok (hal/links api-client :port ports)))
    (continent-not-found id)))

(defnk $:id$regions$GET
  "List all regions on a continent."
  {:responses {200 [Region] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [continent (continents/by-id db id)]
    (let [regions (regions/in-continent db continent query-params)]
      (ok (hal/links api-client :region regions)))
    (continent-not-found id)))

(defnk $:id$spots$GET
  "List all spots on a continent."
  {:responses {200 [Spot] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- SpotsParams]
   [:resources api-client db]]
  (if-let [continent (continents/by-id db id)]
    (let [spots (spots/in-continent db continent query-params)
          spots (enhance-spots api-client db spots query-params)]
      (ok spots))
    (continent-not-found id)))

(defnk $:id$surfers$GET
  "List all users on a continent."
  {:responses {200 [User] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    identity :- (s/maybe User)
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [continent (continents/by-id db id)]
    (->> (users/in-continent db continent query-params)
         (map #(users/safe-user identity %))
         (hal/links api-client :user)
         (ok))
    (continent-not-found id)))

(set! *warn-on-reflection* false)
