(ns burningswell.api.countries
  "The countries of the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.spots :refer [enhance-spots]]
            [burningswell.db.airports :as airports]
            [burningswell.db.countries :as countries]
            [burningswell.db.ports :as ports]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.db.users :as users]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s])
  (:import [java.io ByteArrayInputStream]))

(set! *warn-on-reflection* true)

(defn- country-not-found
  "Return a 404 response for a country that could not be found by `id`."
  [id]
  (not-found (format "Country %s not found" id)))

(defnk $GET
  "List all countries."
  {:responses {200 [Country]}}
  [[:request query-params :- CountriesParams]
   [:resources api-client db]]
  (let [countries (countries/all db query-params)]
    (ok (hal/links api-client :country countries))))

(defnk $POST
  "Create a new country."
  {:responses {201 Country}}
  [[:request body :- create-country]
   [:resources api-client db broker]]
  (let [country (countries/insert db body)
        country (hal/link api-client :country country)]
    (publish broker "countries.created" country)
    (created country)))

(defnk $:id$GET
  "Show a country."
  {:responses {200 Country 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [country (countries/by-id db id)]
    (ok (hal/link api-client :country country))
    (country-not-found id)))

(defnk $:id$DELETE
  "Delete a country."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [country (countries/by-id db id)]
    (do (countries/delete db country)
        (publish broker "countries.deleted" country)
        (no-content))
    (country-not-found id)))

(defnk $:id$PUT
  "Update a country."
  {:responses {200 Country 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-country]
   [:resources api-client db broker]]
  (if-let [country (countries/by-id db id)]
    (let [country (countries/update db (merge country body))
          country (countries/by-id db id)
          country (hal/link api-client :country country)]
      (publish broker "countries.updated" country)
      (ok country))
    (country-not-found id)))

(defnk $:id$airports$GET
  "List all airports on a country."
  {:responses {200 [Airport] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [country (countries/by-id db id)]
    (let [airports (airports/in-country db country query-params)]
      (ok (hal/links api-client :airport airports)))
    (country-not-found id)))

(defnk $:id$image.png$GET
  {:responses
   {200 ByteArrayInputStream
    404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- ImageParams]
   [:resources api-client db]]
  (if-let [country (countries/by-id db id)]
    (ok (ByteArrayInputStream.
         (countries/as-png db country query-params))
        {"cache-control" "public, max-age=86400"
         "content-type" "image/png"})
    (country-not-found id)))

(defnk $:id$ports$GET
  "List all ports on a country."
  {:responses {200 [Port] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [country (countries/by-id db id)]
    (let [ports (ports/in-country db country query-params)]
      (ok (hal/links api-client :port ports)))
    (country-not-found id)))

(defnk $:id$regions$GET
  "List all regions on a country."
  {:responses {200 [Region] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [country (countries/by-id db id)]
    (let [regions (regions/in-country db country query-params)]
      (ok (hal/links api-client :region regions)))
    (country-not-found id)))

(defnk $:id$spots$GET
  "List all spots on a country."
  {:responses {200 [Spot] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- SpotsParams]
   [:resources api-client db]]
  (if-let [country (countries/by-id db id)]
    (let [spots (spots/in-country db country query-params)
          spots (enhance-spots api-client db spots query-params)]
      (ok spots))
    (country-not-found id)))

(defnk $:id$surfers$GET
  "List all users on a country."
  {:responses {200 [User] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    identity :- (s/maybe User)
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [country (countries/by-id db id)]
    (->> (users/in-country db country query-params)
         (map #(users/safe-user identity %))
         (hal/links api-client :user)
         (ok))
    (country-not-found id)))

(set! *warn-on-reflection* false)
