(ns burningswell.api.parser
  (:refer-clojure :exclude [read])
  (:require [burningswell.db.countries :as countries]
            [burningswell.db.search :as search]
            [burningswell.db.spots :as spots]
            [burningswell.db.regions :as regions]
            [burningswell.db.photos :as photos]
            [burningswell.db.users :as users]
            [burningswell.db.weather :as weather]
            [clojure.string :as str]
            [om.next.server :as om]))

(defn- add-pagination
  "Add default :page and :per-page entries to `params`. "
  [{:keys [page per-page] :as params}]
  (cond-> params
    (nil? page)
    (assoc :page 1)
    (nil? per-page)
    (assoc :per-page 10)))

(defmulti read (fn [env key params] key))

(defn- error-no-read-handler [key params]
  (format "No remote handler for read key `%s`. Params: %s"
          key params))

(defmethod read :default
  [env key params]
  (throw (ex-info "No read handler." {:key key} )))

(defmethod read :spots/bounding-box
  [{:keys [db]} _ params]
  (let [{:keys [bounding-box page per-page zoom] :as params}
        (add-pagination params)]
    (when bounding-box
      {:value
       (cond
         (or (nil? zoom) (< zoom 5))
         (->> (assoc params :min-spots 1)
              (countries/all db)
              (countries/assoc-photo db)
              (map #(assoc % :type :country)))
         (< zoom 9)
         (->> (assoc params :min-spots 1)
              (regions/all db)
              (regions/assoc-photo db)
              (map #(assoc % :type :region)))
         :else
         (->> (spots/all db params)
              (spots/assoc-photo db)
              (map #(assoc % :type :spot))))})))

(defmethod read :spots/list
  [{:keys [db]} _ params]
  (let [params (add-pagination params)]
    {:value (vec (map #(dissoc % :location)
                      (->> (spots/all db params)
                           (spots/assoc-photo db))))}))

(defmethod read :pages/country
  [{:keys [db]} _ {:keys [id] :as params}]
  (when-let [country (countries/by-id db id)]
    (let [params (add-pagination params)
          params (assoc params :min-spots 1)
          regions (regions/in-country db country params)
          regions (regions/assoc-photo db regions)]
      {:value (assoc country :regions regions) })))

(defmethod read :pages/region
  [{:keys [db]} _ {:keys [id] :as params}]
  (when-let [region (regions/by-id db id)]
    (let [params (add-pagination params)
          spots (spots/in-region db region params)
          spots (spots/assoc-photo db spots)]
      {:value (assoc region :spots spots)})))

(defmethod read :pages/spot
  [{:keys [db]} _ {:keys [id]}]
  (when-let [spot (spots/by-id db id)]
    (let [spot (first (spots/assoc-photo db [spot]))
          spots (spots/around-spot db spot)
          spots (spots/assoc-photo db spots)]
      {:value (assoc spot :spots spots)})))

(defmethod read :spot/by-id
  [{:keys [db]} _ {:keys [id]}]
  {:value (spots/by-id db id)})

(defmethod read :spots/around
  [{:keys [db]} _ {:keys [id] :as params}]
  (let [params (add-pagination params)]
    (when-let [spot (spots/by-id db id)]
      {:value (spots/around-spot db spot params)})))

(defmethod read :search/results
  [{:keys [db]} _ {:keys [page per-page query] :as params}]
  (let [params (add-pagination params)]
    (when-not (str/blank? query)
      {:value (search/autocomplete db params)})))

(defmethod read :search/details
  [{:keys [db]} _ {:keys [page per-page query] :as params}]
  (let [params (add-pagination params)]
    {:value (search/details db params)}))

(defmethod read :weather/spot
  [{:keys [db]} _ {:keys [id] :as params}]
  (let [params (add-pagination params)
        spot (spots/by-id db id)]
    {:value (weather/weather-by-spot db spot)}))

(defmethod read :photos/spot
  [{:keys [db]} _ {:keys [id] :as params}]
  (let [params (add-pagination params)
        spot (spots/by-id db id)]
    {:value (photos/by-spot db spot {:images true})}))

(defmulti mutate (fn [env key params] key))

(defmethod mutate 'signin/submit
  [{:keys [db]} _ {:keys [username password] :as params}]
  {:action #(users/authenticate-password db params)
   :value {:keys [:signin]}})

(defmethod mutate :default
  [env key params]
  (throw (ex-info "No mutation handler." {:key key} )))

(def parser
  (om/parser {:read read :mutate mutate}))

(comment
  (let [db (-> reloaded.repl/system :db)]
    (clojure.pprint/pprint
     (regions/assoc-photo db [(regions/by-id db 438)]))))
