(ns burningswell.api.ports
  "The ports in the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.db.ports :as ports]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn port-not-found
  "Return a 404 response for a port that could not be found by `id`."
  [id]
  (not-found (format "Port %s not found" id)))

(defnk $GET
  "List all ports."
  {:responses {200 [Port]}}
  [[:request query-params :- PaginationParams]
   [:resources api-client db]]
  (let [ports (ports/all db query-params)]
    (ok (hal/links api-client :port ports))))

(defnk $POST
  "Create a new port."
  {:responses {201 Port}}
  [[:request body :- create-port]
   [:resources api-client db broker]]
  (let [port (ports/insert db body)
        port (hal/link api-client :port port)]
    (publish broker "ports.created" port)
    (created port)))

(defnk $:id$GET
  "Show a port."
  {:responses {200 Port 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [port (ports/by-id db id)]
    (ok (hal/link api-client :port port))
    (port-not-found id)))

(defnk $:id$DELETE
  "Delete a port."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [port (ports/by-id db id)]
    (do (ports/delete db port)
        (publish broker "ports.deleted" port)
        (no-content))
    (port-not-found id)))

(defnk $:id$PUT
  "Update a port."
  {:responses {200 Port 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-port]
   [:resources api-client db broker]]
  (if-let [port (ports/by-id db id)]
    (let [port (ports/update db (merge port body))
          port (hal/link api-client :port port)]
      (publish broker "ports.updated" port)
      (ok port))
    (port-not-found id)))
