(ns burningswell.api.regions
  "The regions of the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.client :as api]
            [burningswell.api.hal :as hal]
            [burningswell.api.schemas :refer :all]
            [burningswell.api.spots :refer [enhance-spots]]
            [burningswell.db.airports :as airports]
            [burningswell.db.ports :as ports]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.db.users :as users]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s])
  (:import [java.awt Color]
           [java.io ByteArrayInputStream]))

(set! *warn-on-reflection* true)

(defn- region-not-found
  "Return a 404 response for a region that could not be found by `id`."
  [id]
  (not-found (format "Region %s not found" id)))

(defnk $GET
  "List all regions."
  {:responses {200 [Region]}}
  [[:request query-params :- RegionsParams]
   [:resources api-client db]]
  (let [regions (regions/all db query-params)]
    (ok (hal/links api-client :region regions))))

(defnk $POST
  "Create a new region."
  {:responses {201 Region}}
  [[:request body :- create-region]
   [:resources api-client db broker]]
  (let [region (regions/insert db body)
        region (hal/link api-client :region region)]
    (publish broker "regions.created" region)
    (created region)))

(defnk $:id$GET
  "Show a region."
  {:responses {200 Region 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [region (regions/by-id db id)]
    (ok (hal/link api-client :region region))
    (region-not-found id)))

(defnk $:id$DELETE
  "Delete a region."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [region (regions/by-id db id)]
    (do (regions/delete db region)
        (publish broker "regions.deleted" region)
        (no-content))
    (region-not-found id)))

(defnk $:id$PUT
  "Update a region."
  {:responses {200 Region 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-region]
   [:resources api-client db broker]]
  (if-let [region (regions/by-id db id)]
    (let [region (regions/update db (merge region body))
          region (hal/link api-client :region region)]
      (publish broker "regions.updated" region)
      (ok region))
    (region-not-found id)))

(defnk $:id$airports$GET
  "List all airports in a region."
  {:responses {200 [Airport] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [region (regions/by-id db id)]
    (let [airports (airports/in-region db region query-params)]
      (ok (hal/links api-client :airport airports)))
    (region-not-found id)))

(defnk $:id$image.png$GET
  {:responses
   {200 ByteArrayInputStream
    404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- ImageParams]
   [:resources api-client db]]
  (if-let [region (regions/by-id db id)]
    (ok (ByteArrayInputStream.
         (regions/as-png db region query-params))
        {"cache-control" "public, max-age=86400"
         "content-type" "image/png"})
    (region-not-found id)))

(defnk $:id$ports$GET
  "List all ports in a region."
  {:responses {200 [Port] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [region (regions/by-id db id)]
    (let [ports (ports/in-region db region query-params)]
      (ok (hal/links api-client :port ports)))
    (region-not-found id)))

(defnk $:id$spots$GET
  "List all spots in a region."
  {:responses {200 [Spot] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- SpotsParams]
   [:resources api-client db]]
  (if-let [region (regions/by-id db id)]
    (let [spots (spots/in-region db region query-params)
          spots (enhance-spots api-client db spots query-params)]
      (ok spots))
    (region-not-found id)))

(defnk $:id$surfers$GET
  "List all users in a region."
  {:responses {200 [User] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    identity :- (s/maybe User)
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [region (regions/by-id db id)]
    (->> (users/in-region db region query-params)
         (map #(users/safe-user identity %))
         (hal/links api-client :user)
         (ok))
    (region-not-found id)))

(set! *warn-on-reflection* false)
