(ns burningswell.api.ring
  "A standard set of commonly used ring middleware"
  (:require [burningswell.api.schemas :refer [explain-errors]]
            [burningswell.json :as json]
            [burningswell.transit :as transit]
            [ring.util.http-status :as status]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [geo.json]
            [hiccup.element :refer [javascript-tag]]
            [hiccup.page :refer [html5 include-css include-js]]
            [plumbing.core :refer :all]
            [ring.middleware.content-type :refer [wrap-content-type]]
            [ring.middleware.resource :refer [wrap-resource]]
            [ring.middleware.not-modified :refer [wrap-not-modified]]
            [ring.middleware.cors :as cors]
            [ring.middleware.defaults :as defaults]
            [ring.middleware.params :as params]
            [ring.util.response :refer [content-type get-header]]
            [schema.core :as s]))

(def highlight-css
  "The highlight.js stylesheet."
  (str "//cdnjs.cloudflare.com/ajax/libs/highlight.js/9.2.0"
       "/styles/solarized-light.min.css"))

(def highlight-js
  "The highlight.js JavaScript code."
  "//cdnjs.cloudflare.com/ajax/libs/highlight.js/9.2.0/highlight.min.js")

(defn- transit-request? [request]
  (if-let [type (:content-type request)]
    (not (empty? (re-find #"^application/(.+\+)?json" type)))))

(defn wrap-cors
  "Wrap `handler` to return CORS headers."
  [handler]
  (cors/wrap-cors
   handler
   :access-control-allow-credentials "true"
   :access-control-allow-headers #{"accept" "authorization" "content-type"}
   :access-control-allow-methods #{:get :delete :head :post :patch :put}
   :access-control-allow-origin #".*"
   :access-control-expose-headers
   (str/join ", " ["etag" "link" "x-total"])))

(defn replace-links
  "Replace all URLs in `s` with an HTML anchor."
  [s]
  (str/replace s #"\"(https?://[^\"]+)\"" "\"<a href=\"$1\">$1</a>\""))

(defn fade-in-key-frames [from to]
  (format
   "@keyframes fadein {
     from { opacity: %d; }
     to { opacity: %d; }
    }" from to))

(defn pprint-html [response]
  (replace-links
   (html5
    [:head
     (include-css "/stylesheets/api.css")
     (include-css highlight-css)
     (include-js highlight-js)
     (javascript-tag "hljs.initHighlightingOnLoad();")]
    [:body
     [:h3 (:status response) " - " (status/get-description (:status response))]
     [:pre [:code (with-out-str (pprint (:body response)))]]])))

(defn wrap-html-response
  "Return a Ring handler that converts the body of the response to
  HTML format."
  [handler & {:as opts}]
  (fn [request]
    (let [response (handler request)]
      (if (and (coll? (:body response))
               (or (:default opts)
                   (re-find #"text/html"
                            (str (get-header request "accept")))))
        (-> (assoc response :body (pprint-html response))
            (content-type "text/html"))
        response))))

(defn wrap-json-response
  "Return a Ring handler that converts the body of the response to
  JSON format."
  [handler & {:as opts}]
  (fn [request]
    (let [response (handler request)]
      (if (and (coll? (:body response))
               (or (:default opts)
                   (re-matches #"application/json"
                               (str (get-header request "accept")))))
        (-> (assoc response :body (json/json-str (:body response)))
            (content-type "application/json"))
        response))))

(defn wrap-transit-request
  "Return a Ring handler that converts the body of the request from
  Transit format into a Clojure data structure."
  [handler]
  (fn [request]
    (if (and (:body request)
             (transit-request? request))
      (let [body (transit/decode (:body request))]
        (handler (assoc request :body body)))
      (handler request))))

(defn wrap-transit-response
  "Return a Ring handler that converts the body of the response to
  Transit format."
  [handler & {:as opts}]
  (fn [request]
    (let [response (handler request)]
      (if (and (coll? (:body response))
               (or (:default opts)
                   (re-matches #"application/transit\+json"
                               (str (get-header request "accept")))))
        (-> (assoc response :body (transit/encode (:body response)))
            (content-type "application/transit+json"))
        response))))

(defn wrap-slurp-request
  [handler]
  (fn [{:keys [body] :as request}]
    (let [body (if body (slurp body))]
      (handler (assoc request :body body)))))

(defn wrap-unprocessable-entity
  "Return a handler that catches :coercion-error exceptions and
  returns a HTTP status code 422 (unprocessable entity) response."
  [handler]
  (fn [request]
    (try (handler request)
         (catch Exception e
           (let [{:keys [context data schema type error value]} (ex-data e)]
             (case type
               :schema.core/error
               {:status 422
                :body {:data value
                       :errors (explain-errors error)
                       :schema (s/explain schema)}}
               :coercion-error
               {:status 422
                :body {:data data
                       :errors (explain-errors (s/check schema data))
                       :schema (s/explain schema)}}
               (throw e)))))))

(defn wrap-request-logging
  [handler & {:as opts}]
  (fn [request]
    (let [response (handler request)]
      (log/infof "method=%s, uri=%s"
                 (name (:request-method request))
                 (:uri request))
      response)))

(defn wrap-exception [f]
  (fn [request]
    (try (f request)
         (catch Exception e
           (.printStackTrace e)
           {:status 500
            :body "Exception caught"}))))

(defn keywordize-middleware [handler]
  (fn [req]
    (handler
     (update-in req [:query-params] keywordize-map))))

(defn api-defaults
  "Return the ring-defaults settings for the API."
  [resources]
  (let [config (:config resources)]
    (-> (assoc defaults/secure-api-defaults :proxy true)
        (assoc-in [:security :ssl-redirect] (:ssl-redirect? config)))))

(defn ring-middleware
  [handler resources]
  (-> handler
      (keywordize-middleware)
      (params/wrap-params)
      (wrap-unprocessable-entity)
      (wrap-html-response)
      (wrap-json-response)
      (wrap-transit-response :default true)
      (wrap-transit-request)
      (wrap-slurp-request)
      (wrap-resource "public")
      (wrap-content-type)
      (wrap-not-modified)
      (defaults/wrap-defaults (api-defaults resources))
      (wrap-cors)
      (wrap-request-logging)
      (wrap-exception)))
