(ns burningswell.api.spots
  "The surf spots of the world."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.hal :as hal]
            [burningswell.api.photos :refer [make-photo]]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.comments :as comments]
            [burningswell.db.countries :as countries]
            [burningswell.db.photos :as photos]
            [burningswell.db.ratings :as ratings]
            [burningswell.db.regions :as regions]
            [burningswell.db.sessions :as sessions]
            [burningswell.db.spots :as spots]
            [burningswell.db.time-zones :as time-zones]
            [burningswell.db.weather :as weather]
            [burningswell.http.response :refer [created ok]]
            [clj-time.coerce :refer [to-date-time to-long]]
            [clj-time.core :refer [hour]]
            [hiccup.core :refer [html]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(defn make-spot [api-client spot]
  (-> (hal/link api-client :spot spot)
      (update-in [:_embedded :photo]
                 #(make-photo api-client %))))

(defn enhance-spots
  "Enhance `spots` with a photo and the current weather."
  [api-client db spots & [opts]]
  (let [spots (spots/assoc-photo db spots)
        spots (spots/assoc-current-weather db spots opts)]
    (map #(make-spot api-client %) spots)))

(defn enhance-spot [api-client db spot & [opts]]
  (first (enhance-spots api-client db [spot] opts)))

(defn- spot-not-found
  "Return a 404 response for a spot that could not be found by `id`."
  [id]
  (not-found (format "Spot %s not found" id)))

(defnk $GET
  "List all surf spots."
  {:responses {200 [Spot]}}
  [[:request query-params :- SpotsParams]
   [:resources api-client db]]
  (let [spots (spots/all db query-params)
        spots (enhance-spots api-client db spots query-params)]
    (ok spots)))

(defnk $:id$GET
  "Show a surf spot."
  {:responses {200 Spot 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- SpotParams]
   [:resources api-client db]]
  (if-let [spot (spots/by-id db id)]
    (ok (enhance-spot api-client db spot query-params))
    (spot-not-found id)))

(defnk $POST
  "Create a new spot."
  {:responses {201 Spot}}
  [[:request
    body :- create-spot
    identity :- User]
   [:resources api-client db broker]]
  (let [location (:location body)
        body (-> (update-in
                  body [:_embedded :country :id]
                  #(or % (:id (countries/closest db location))))
                 (update-in [:_embedded :region :id]
                            #(or % (:id (regions/closest db location))))
                 (assoc-in [:_embedded :user] identity)
                 (assoc-in [:_embedded :time-zone]
                           (time-zones/by-location db location)))
        spot (spots/insert db body)
        spot (enhance-spot api-client db spot)]
    (publish broker "spots.created" spot)
    (created spot)))

(defnk $:id$DELETE
  "Delete a surf spot."
  {:responses {204 s/Any 403 Forbidden 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources broker db]]
  (if-let [spot (spots/by-id db id)]
    (do (spots/delete db spot)
        (publish broker "spots.deleted" spot)
        (no-content))
    (spot-not-found id)))

(defnk $:id$PUT
  "Update a region."
  {:responses {200 Spot 403 Forbidden 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-spot
    identity :- User]
   [:resources api-client db broker]]
  (if-let [spot (spots/by-id db id)]
    (let [body (-> (assoc body :id id)
                   (assoc-in [:_embedded :user] identity))
          spot (spots/update db body)
          spot (enhance-spot api-client db spot)]
      (publish broker "regions.updated" spot)
      (ok spot))
    (spot-not-found id)))

(defnk $:id$spots-around$GET
  "Return spots around the current spot."
  {:responses {200 [Spot] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- SpotsParams]
   [:resources api-client db]]
  (if-let [spot (spots/by-id db id)]
    (let [spots (spots/around-spot db spot query-params)
          spots (enhance-spots api-client db spots query-params)]
      (ok (remove #(= (:id %) (:id spot)) spots)))
    (spot-not-found id)))

(defnk $:id$comments$GET
  "List all comments of the spot."
  {:responses {200 [Comment] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [spot (spots/by-id db id)]
    (let [comments (comments/by-spot db spot query-params)
          comments (hal/links api-client :comment comments)]
      (ok comments))
    (spot-not-found id)))

(defnk $:id$photos$GET
  "List all photos of the spot."
  {:responses {200 [Photo] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources api-client db]]
  (if-let [spot (spots/by-id db id)]
    (let [photos (photos/by-spots db [spot] query-params)]
      (ok (map #(make-photo api-client %) photos)))
    (spot-not-found id)))

(defnk $:id$ratings$GET
  "List the ratings of a spot."
  {:responses {200 [Rating] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db]]
  (if-let [spot (spots/by-id db id)]
    (let [ratings (ratings/ratings-by-spot db spot query-params)]
      (ok ratings))
    (spot-not-found id)))

(defnk $:id$sessions$GET
  "List the sessions of a spot."
  {:responses {200 [Session] 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db]]
  (if-let [spot (spots/by-id db id)]
    (let [sessions (sessions/sessions-by-spot db spot query-params)]
      (ok sessions))
    (spot-not-found id)))

(defnk $:id$weather$GET
  "Show the the weather at a spot."
  {:responses {200 Weather 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- WeatherParams]
   [:resources db]]
  (if-let [spot (spots/by-id db id)]
    (let [weather (weather/weather-by-spot db spot query-params)]
      (ok weather {"cache-control" "public, max-age=86400"}))))

(defn render-bar-chart
  [data]
  (let [timestamps (map :valid-time data)
        fill-color "#444444"
        values (map :value data)
        max-value (apply max (remove nil? values))
        max-width 500
        max-height 89
        padding 1.0
        width (/ max-width (count values))]
    (str
     "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\" "
     "\"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">"
     (html
      [:svg {:class "wave-heights-chart"
             :version "1.1"
             :preserveAspectRatio "none"
             :viewBox (format "0 0 %s %s" max-width max-height)
             :xmlns "http://www.w3.org/2000/svg"}
       [:g
        (map-indexed
         (fn [index data]
           (let [value (:value data)
                 timestamp (:valid-time data)
                 height (if value (/ (* max-height value) max-value) 1)
                 hours (hour (to-date-time timestamp))
                 class (if (and (>= hours 5) (<= hours 22)) "day" "night")]
             [:g
              {:class "wave-heights-chart__day"
               :data-time (to-long timestamp)
               :data-value value}
              [:rect
               {:class "wave-heights-chart__day-bg"
                :height max-height
                :style "fill: #cccccc; fill-opacity: 0;"
                :width (- width padding)
                :x (+ (* index width) padding)
                :y 0}]
              [:rect
               {:class "wave-heights-chart__day-height"
                :height height
                :style (str "fill: " fill-color ";")
                :width (- width padding)
                :x (+ (* index width) padding)
                :y (- max-height height)}]]))
         data)]]))))

(defnk $:id$wave-heights.svg$GET
  "Return the wave heights SVG chart."
  {:responses {200 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- WaveHeightParams]
   [:resources db]]
  (if-let [spot (spots/by-id db id)]
    (let [wave-heights (weather/wave-heights-by-spot db spot query-params)]
      (if-not (empty? wave-heights)
        (ok (render-bar-chart wave-heights)
            {"cache-control" "public, max-age=86400"
             "content-type" "image/svg+xml"})
        (not-found "No wave height available.")))
    (spot-not-found id)))
