(ns burningswell.api.weather.datasets
  "The weather datasets."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.weather :as weather]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn- dataset-not-found
  "Return a 404 response for a region that could not be found by `id`."
  [id]
  (not-found (format "Weather dataset %s not found" id)))

(defnk $GET
  "List all weather datasets."
  {:responses {200 [WeatherDataset]}}
  [[:request query-params :- PaginationParams]
   [:resources db]]
  (let [datasets (weather/datasets db query-params)]
    (ok datasets)))

(defnk $POST
  "Create a new weather dataset."
  {:responses {201 WeatherDataset}}
  [[:request body :- create-weather-dataset]
   [:resources db broker]]
  (let [dataset (weather/create-dataset db body)]
    (publish broker "weather.datasets.created" dataset)
    (created dataset)))

(defnk $:id$GET
  "Show a weather dataset."
  {:responses {200 WeatherDataset 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db]]
  (if-let [dataset (weather/dataset-by-id db id)]
    (ok dataset)
    (dataset-not-found id)))

(defnk $:id$DELETE
  "Delete a weather dataset."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [dataset (weather/dataset-by-id db id)]
    (do (weather/delete-dataset db dataset)
        (publish broker "weather.datasets.deleted" dataset)
        (no-content))
    (dataset-not-found id)))

(defnk $:id$PUT
  "Update a weather dataset."
  {:responses {200 WeatherDataset 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-weather-dataset]
   [:resources db broker]]
  (if-let [dataset (weather/dataset-by-id db id)]
    (let [dataset (weather/update-dataset db (merge dataset body))]
      (publish broker "weather.datasets.updated" dataset)
      (ok dataset))
    (dataset-not-found id)))

(set! *warn-on-reflection* false)
