(ns burningswell.api.weather.models
  "The weather models."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.weather :as weather]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn- model-not-found
  "Return a 404 response for a region that could not be found by `id`."
  [id]
  (not-found (format "Weather model %s not found" id)))

(defnk $GET
  "List all weather models."
  {:responses {200 [WeatherModel]}}
  [[:request query-params :- PaginationParams]
   [:resources db]]
  (let [models (weather/models db query-params)]
    (ok models)))

(defnk $POST
  "Create a new weather model."
  {:responses {201 WeatherModel}}
  [[:request body :- create-weather-model]
   [:resources db broker]]
  (let [model (weather/create-model db body)]
    (publish broker "weather.models.created" model)
    (created model)))

(defnk $:id$GET
  "Show a weather model."
  {:responses {200 WeatherModel 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db]]
  (if-let [model (weather/model-by-id db id)]
    (ok model)
    (model-not-found id)))

(defnk $:id$DELETE
  "Delete a weather model."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [model (weather/model-by-id db id)]
    (do (weather/delete-model db model)
        (publish broker "weather.models.deleted" model)
        (no-content))
    (model-not-found id)))

(defnk $:id$PUT
  "Update a weather model."
  {:responses {200 WeatherModel 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-weather-model]
   [:resources db broker]]
  (if-let [model (weather/model-by-id db id)]
    (let [model (weather/update-model db (merge model body))]
      (publish broker "weather.models.updated" model)
      (ok model))
    (model-not-found id)))

(set! *warn-on-reflection* false)
