(ns burningswell.api.weather.variables
  "The weather variables."
  (:require [burningswell.api.core :refer :all]
            [burningswell.api.schemas :refer :all]
            [burningswell.db.weather :as weather]
            [burningswell.http.response :refer [created ok]]
            [plumbing.core :refer :all]
            [schema.core :as s]))

(set! *warn-on-reflection* true)

(defn- variable-not-found
  "Return a 404 response for a region that could not be found by `id`."
  [id]
  (not-found (format "Weather variable %s not found" id)))

(defnk $GET
  "List all weather variables."
  {:responses {200 [WeatherVariable]}}
  [[:request query-params :- PaginationParams]
   [:resources db]]
  (let [variables (weather/variables db query-params)]
    (ok variables)))

(defnk $POST
  "Create a new weather variable."
  {:responses {201 WeatherVariable}}
  [[:request body :- create-weather-variable]
   [:resources db broker]]
  (let [variable (weather/create-variable db body)]
    (publish broker "weather.variables.created" variable)
    (created variable)))

(defnk $:id$GET
  "Show a weather variable."
  {:responses {200 WeatherVariable 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db]]
  (if-let [variable (weather/variable-by-id db id)]
    (ok variable)
    (variable-not-found id)))

(defnk $:id$DELETE
  "Delete a weather variable."
  {:responses {204 s/Any 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    query-params :- PaginationParams]
   [:resources db broker]]
  (if-let [variable (weather/variable-by-id db id)]
    (do (weather/delete-variable db variable)
        (publish broker "weather.variables.deleted" variable)
        (no-content))
    (variable-not-found id)))

(defnk $:id$PUT
  "Update a weather variable."
  {:responses {200 WeatherVariable 404 NotFound}}
  [[:request
    [:uri-args id :- Id]
    body :- create-weather-variable]
   [:resources db broker]]
  (if-let [variable (weather/variable-by-id db id)]
    (let [variable (weather/update-variable db (merge variable body))]
      (publish broker "weather.variables.updated" variable)
      (ok variable))
    (variable-not-found id)))

(set! *warn-on-reflection* false)
