(ns burningswell.api.addresses
  (:require [burningswell.api.core :as core]
            [burningswell.api.middleware.conform :as conform]
            [burningswell.api.middleware.identifier :as identifier]
            [burningswell.api.specs :as specs]
            [burningswell.api.validation :as v]
            [burningswell.db.addresses :as addresses]
            [claro.data :as data]
            [clojure.spec.alpha :as s]))

(s/def :burningswell.api.addresses/id
  (s/and ::specs/id (v/db-id-exists? :addresses)))

(s/def :burningswell.api.addresses/except
  (s/nilable (s/coll-of :burningswell.api.addresses/id)))

(s/def :burningswell.api.addresses/countries
  (s/nilable (s/coll-of :burningswell.api.countries/id)))

(s/def :burningswell.api.addresses/regions
  (s/nilable (s/coll-of :burningswell.api.regions/id)))

(s/def :burningswell.api.addresses/params
  (s/keys :opt-un [:burningswell.api.addresses/countries
                   :burningswell.api.addresses/except
                   :burningswell.api.addresses/regions
                   :burningswell.api.addresses/sort
                   :burningswell.api.pagination/after
                   :burningswell.api.pagination/before
                   :burningswell.api.pagination/first
                   :burningswell.api.pagination/last
                   :burningswell.api.search/query
                   :burningswell.api.specs/direction
                   :burningswell.api.specs/distance
                   :burningswell.api.specs/location]))

(defrecord Address [id name country region]
  conform/Params
  (conform [params env]
    (s/keys :req-un [:burningswell.api.specs/id]))

  identifier/Identifier
  (identifier [_ _]
    {:type :address
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} addresses]
    (addresses/select-batch db addresses)))

(defrecord AddressByLocation [location]
  identifier/Identifier
  (identifier [_ _]
    {:type :address
     :columns [:id]})

  data/Resolvable
  data/BatchedResolvable
  (resolve-batch! [_ {:keys [db]} addresses]
    (addresses/by-locations db (map :location addresses))))

(defrecord Addresses [after before countries distance direction except
                      first last location regions sort query]
  conform/Params
  (conform [params env]
    :burningswell.api.addresses/params)

  data/Resolvable
  (resolve! [params {:keys [db]}]
    (->> {:countries countries
          :direction direction
          :distance distance
          :except except
          :limit (core/limit params)
          :location location
          :offset (core/offset params)
          :query query
          :regions regions
          :sort sort}
         (addresses/search db))))
