(ns burningswell.api.core
  (:require [clojure.edn :as edn]
            [clojure.spec.gen.alpha :as gen]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [datumbazo.core :as sql]
            [datumbazo.postgresql.associations]
            [potemkin :refer [import-vars]]
            [ring.util.codec :as codec]
            [sqlingvo.compiler :as compiler]))

(def ^:dynamic *env* {})

(defmacro with-env [env & body]
  `(binding [*env* ~env] ~@body))

(defn conform [env spec data]
  (with-env env
    (s/conform spec data)))

(defn conform!
  ([spec data]
   (let [result (s/conform spec data)]
     (if (= result ::s/invalid)
       (throw (ex-info "Can't conform data to spec."
                       (s/explain-data spec data)))
       result)))
  ([env spec data]
   (with-env env
     (conform! spec data))))

(defn exercise [env spec]
  (with-env env
    (doall (s/exercise spec))))

(defn explain [env spec data]
  (with-env env
    (s/explain spec data)))

(defn explain-data [env spec data]
  (with-env env
    (doall (s/explain-data spec data))))

(defn valid? [env spec data]
  (with-env env (s/valid? spec data)))

(defn generate [env gen]
  (with-env env
    (gen/generate gen)))

(defn sample [env gen]
  (with-env env
    (doall (gen/sample gen))))

(defn limit [params]
  (or (:first params) (-> params :after :limit) 10))

(defn offset [params]
  (or (-> params :after :offset) 0))

(defn base64-encode
  "Encode `x` into Base64."
  [x]
  (->> (pr-str x)
       (.getBytes)
       (codec/base64-encode)))

(defn base64-decode
  "Decode `x` from Base64."
  [s]
  (some-> (codec/base64-decode s)
          (String. "UTF-8")
          (edn/read-string)))

(defn add-id [m type]
  (->> (base64-encode {:id (:id m) :type type})
       (assoc m :id)))

(defn parse-id [s]
  (edn/read-string (String. (codec/base64-decode s))))

(defn rand-uuid
  "Returns a random UUID."
  []
  (java.util.UUID/randomUUID))

(defn paginate [page per-page]
  (sql/paginate (or page 1) (or per-page 10)))

(defn sort-batch [sorted key resources]
  (map (zipmap (map key resources) resources)
       (map key sorted)))

(defn order-by [table & [{:keys [direction nulls sort]}]]
  (when sort
    (let [column (keyword (format "%s.%s" (name table) (name sort)))
          direction (symbol (name direction))]
      `(~'order-by (~'nulls (~direction ~column) ~(or nulls :last))))))

(defn connection [xs]
  {:edges (map #(hash-map :node %) xs)})

;; Resolve db relationships

(defn- array-paginate-idx
  "Calculate the array (1-based) pagination lower and upper bounds."
  [page per-page]
  (let [page (or page 1)
        per-page (or per-page 10)
        start (inc (* (dec page) per-page))
        end (dec (+ start per-page))]
    [start end]))

(defn- array-paginate
  "Returns the array pagination expression."
  [column {:keys [after before first last order-by]}]
  (let [start (or (-> after :offset) 0)
        end (+ start (or (-> after :limit) (or first 10)))]
    (sql/as `(subvec (array_agg ~column ~order-by) ~start ~end)
            :array-paginate)))

(defn- array-paginate-extract
  "Extract the array pagination results from `rows` and
  adds :total-count to the meta data of each collection."
  [column rows]
  (mapv #(with-meta (mapv (fn [value] {column value}) (:array-paginate %))
           {:total-count (or (:count %) 0)})
        rows))

(defn source
  [alias rows & [{:keys [column type]}]]
  (sql/from
   (sql/as
    (sql/values
     (for [[index row] (map-indexed vector rows)]
       ;; Cast id to type or UUID, because id could be NULL
       [`(cast ~(get row (or column :id)) ~(or type :int)) index]))
    alias [(or column :id) :index])))

(defn- resolve-batch
  "Resolve a batch of `rows` from `table`."
  [db table rows & [{:keys [column columns type where join-column]}]]
  {:pre [(sql/db? db) (keyword? table)]}
  (let [column (or column :id)
        columns (or columns [(keyword (str (name table) ".*"))])
        join-src (keyword (str (name table) "."
                               (name (or join-column column))))]
    (->> @(sql/select db columns
            (source :source rows {:column column :type type})
            (sql/join join-src (keyword (format "source.%s" (name column)))
                      :type :left)
            (when where (sql/where where :and))
            (sql/order-by :source.index))
         (map (fn [row] (when (get row column) row))))))

(defn resolve-by-id
  "Resolve a batch of `rows` from `table`."
  [db table rows & [opts]]
  (resolve-batch db table rows (assoc opts :column :id)))

(s/fdef resolve-by-id
  :args (s/cat :db sql/db?
               :table keyword?
               :rows (s/coll-of map?)
               :opts (s/* (s/nilable map?))))

(defn has-many-sql
  "Batch paginate the has-many relationships of the given `rows` using
  `table`, `foreign-key` and `target-key`."
  [db table foreign-key rows & [{:keys [after before first last
                                        target-key where type
                                        order-by]
                                 :as opts}]]
  {:pre [(sql/db? db)
         (keyword? table)
         (keyword? foreign-key)]}
  (let [type (or type :int)
        target-column (keyword (str "target." (name foreign-key)))
        source-column (keyword (str "source.id")) ]
    (sql/with db [:target
                  (sql/select db [foreign-key
                                  (array-paginate :id opts)
                                  '(count :id)]
                    (sql/from table)
                    (sql/where `(in ~foreign-key ~(map :id rows)))
                    (when where (sql/where where :and))
                    (sql/group-by foreign-key))]
      (sql/select db [:target.*]
        (source :source rows {:type type})
        (sql/join :target
                  `(on (= (cast ~target-column ~type)
                          (cast ~source-column ~type)))
                  :type :left)
        (sql/order-by :source.index)))))

(defn has-many
  [db table foreign-key rows & [opts]]
  (->> @(has-many-sql db table foreign-key rows opts)
       (array-paginate-extract :id)))

(s/fdef has-many
  :args (s/cat :db sql/db?
               :table keyword?
               :foreign-key keyword?
               :rows (s/coll-of map?)))

;; (defmethod compiler/compile-fn :array_agg [db node]
;;   (let [[_ column order-by] (:children node)]
;;     (compiler/concat-sql
;;      "array_agg(" (compiler/compile-sql db column)
;;      (when-not (empty? (:children order-by))
;;        (compiler/concat-sql
;;         " ORDER BY "
;;         (->> (for [child (:children order-by)
;;                    :let [column (-> child :children first)]]
;;                (compiler/concat-sql
;;                 (compiler/compile-sql db column)
;;                 (when-let [direction (-> child :children second)]
;;                   (str " " (-> direction :form name str/upper-case)))))
;;              (compiler/join-sql ", ")))) ")")))

;; (defmethod compiler/compile-fn :array_agg_distinct [db node]
;;   (let [[_ column order-by] (:children node)]
;;     (compiler/concat-sql
;;      "array_agg(DISTINCT " (compiler/compile-sql db column)
;;      (when-not (empty? (:children order-by))
;;        (compiler/concat-sql
;;         " ORDER BY "
;;         (->> (for [child (:children order-by)
;;                    :let [column (-> child :children first)]]
;;                (compiler/concat-sql
;;                 (compiler/compile-sql db column)
;;                 (when-let [direction (-> child :children second)]
;;                   (str " " (-> direction :form name str/upper-case)))))
;;              (compiler/join-sql ", ")))) ")")))

;; ;; Access a slice of an array with SQLingvo.

;; (defmethod compiler/compile-fn :subvec [db node]
;;   (let [[_ array start end] (:children node)]
;;     (compiler/concat-sql
;;      "(" (compiler/compile-sql db array)
;;      (str ")[" (:val start) ":" (:val end) "]"))))
