(ns burningswell.api.middleware.conform
  (:require [burningswell.api.core :as core]
            [burningswell.api.errors :as errors]
            [claro.data :as data]
            [claro.engine :as engine]
            [claro.runtime.impl :as impl]
            [clojure.spec.alpha :as s]
            [potemkin :refer [defprotocol+]]
            [taoensso.timbre :as log]))

(defprotocol+ Params
  (conform [resolvable env]))

(extend-type Object
  Params
  (conform [resolvable env]
    nil))

(defn explain-data [env spec resolveable]
  (let [data (core/explain-data env spec resolveable)]
    (->> {:explanation (with-out-str (core/explain env spec resolveable))
          :messages (-> (errors/update-problems env data)
                        ::s/problems errors/messages)
          :resolveable (-> resolveable class .getName)
          :spec (s/form spec)}
         ;; Remove the spec key, since it can contain reify, which is
         ;; not JSON serializable.
         (merge (dissoc data :clojure.spec.alpha/spec)))))

(defn- conform-resolveable
  "Resolve the spec for `resolveable` and conform the `resolveable`
  against it. Returns either the conformed resolvable or the problems
  found wrapped in a `data/error` container."
  [env resolveable]
  (if-let [spec (conform resolveable env)]
    (let [result (core/conform env spec resolveable)]
      (if (s/invalid? result)
        (let [errors (explain-data env spec resolveable)]
          (log/debug {:msg "Invalid parameters provided"
                      :errors errors
                      :resolvable resolveable})
          (data/error "Invalid parameters provided" errors))
        (merge resolveable result)))
    resolveable))

(defn- conform-resolvables
  [env batch]
  (->> (for [resolvable batch]
         [resolvable (conform-resolveable env resolvable)])
       (into {})))

(defn- wrap-conform-params*
  [impl resolver]
  (fn [env batch]
    (let [resolvable->conformed (conform-resolvables env batch)
          pending-resolvables (remove data/error? (vals resolvable->conformed))]
      (if (empty? pending-resolvables)
        resolvable->conformed
        (impl/chain1
         impl
         (resolver env pending-resolvables)
         (fn [result]
           (->> (for [[resolvable conformed] resolvable->conformed
                      :when (contains? result conformed)]
                  [resolvable (get result conformed)])
                (into {}))))))))

(defn wrap-conform-params
  [engine]
  (let [impl (engine/impl engine)]
    (engine/wrap engine #(wrap-conform-params* impl %))))
