(ns burningswell.api.oauth.callback
  (:require [burningswell.api.core :as core]
            [burningswell.api.jwt :as jwt]
            [burningswell.api.middleware.commands :as commands]
            [burningswell.api.middleware.events :as events]
            [burningswell.api.oauth.providers :as providers]
            [burningswell.api.oauth.users :as users]
            [burningswell.routes :as routes]
            [clj-http.client :as http]
            [inflections.core :as infl]
            [no.en.core :refer [format-url parse-url]]
            [oauth.one :as one]
            [ring.util.response :as response]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log]))

(def callback-namespace
  "burningswell.api.oauth.callback")

;; Events

(defn- provider-name
  "Returns the provider name from the :route-params of `request`."
  [request]
  (-> request :route-params :name))

(defn- provider
  "Returns the OAuth provider for `request`."
  [env request]
  (some->> (provider-name request)
           (providers/by-name env)))

(defn- access-token-url
  "Returns the parsed access-token url of `provider`."
  [provider]
  (-> provider :access-token-url parse-url))

(defn- state
  "Returns the state from `request`."
  [{:keys [jwt]} request]
  (try (jwt/unsign jwt (-> request :params :state))
       (catch Throwable e nil)))

(defn- valid-state?
  "Returns true if the state parameter of `request` is valid, otherwise false.."
  [env request]
  (= (:remote-addr (state env request))
     (:remote-addr request)))

(defn validate-state!
  "Returns the state from `request`."
  [env request]
  (when-not (valid-state? env request)
    (throw (ex-info "Invalid OAuth state parameter."
                    {:type ::invalid-state-error
                     :state (state env request)}))))

;; Request access token

(defmulti access-token-request
  "Returns the request for `provider` to obtain an access token."
  (fn [env provider request] (-> provider :name keyword)))

(defmethod access-token-request :facebook [env provider request]
  (merge (access-token-url provider)
         {:as :auto
          :coerce :always
          :method :post
          :throw-exceptions false
          :query-params
          {:client_id (-> env :facebook :client-id)
           :client_secret (-> env :facebook :client-secret)
           :code (-> request :params :code)
           :redirect_uri (:callback-url provider)}}))

(defmethod access-token-request :google [env provider request]
  (merge (access-token-url provider)
         {:as :auto
          :coerce :always
          :method :post
          :throw-exceptions false
          :query-params
          {:client_id (-> env :google :client-id)
           :client_secret (-> env :google :client-secret)
           :code (-> request :params :code)
           :grant_type "authorization_code"
           :redirect_uri (:callback-url provider)}}))

(defmethod access-token-request :linkedin [env provider request]
  (merge (access-token-url provider)
         {:as :auto
          :coerce :always
          :method :post
          :throw-exceptions false
          :query-params
          {:client_id (-> env :linkedin :client-id)
           :client_secret (-> env :linkedin :client-secret)
           :code (-> request :params :code)
           :grant_type "authorization_code"
           :redirect_uri (:callback-url provider)}}))

(defmethod access-token-request :twitter [env provider request]
  (let [consumer (providers/oauth-v1-consumer env provider)]
    (-> (one/access-token-request
         consumer {"oauth_token" (-> request :params :oauth_token)
                   "oauth_verifier" (-> request :params :oauth_verifier)})
        (assoc :as :x-www-form-urlencoded))))

(defn request-access-token!
  "Request the OAuth access token for `provider`."
  [env command provider request]
  (let [request (access-token-request env provider request)
        {:keys [status body] :as response} (http/request request)]
    (if (= status 200)
      (infl/hyphenate-keys body)
      (throw (ex-info "Can't request OAuth access token."
                      {:type ::request-access-token-error
                       :provider provider
                       :response response})))))

;; Request profile

(defmulti profile-request
  "Returns the clj-http request to fetch the profile from `provider`."
  (fn [env provider token] (-> provider :name keyword)))

(defmethod profile-request :facebook [env provider token]
  {:as :json
   :coerce :always
   :method :get
   :oauth-token (:access-token token)
   :throw-exceptions false
   :query-params {:fields "id,first_name,last_name,email"}
   :scheme :https
   :server-name "graph.facebook.com"
   :uri "/v2.6/me"})

(defmethod profile-request :google [env provider token]
  {:as :auto
   :coerce :always
   :method :get
   :oauth-token (:access-token token)
   :throw-exceptions false
   :query-params {:scope "email profile"}
   :scheme :https
   :server-name "www.googleapis.com"
   :uri "/plus/v1/people/me"})

(defmethod profile-request :linkedin [env provider token]
  {:as :auto
   :coerce :always
   :method :get
   :oauth-token (:access-token token)
   :throw-exceptions false
   :query-params {:format "json"}
   :scheme :https
   :server-name "api.linkedin.com"
   :uri "/v1/people/~:(id,first-name,last-name,email-address,picture-url)"})

;; https://api.twitter.com/account/verify_credentials.json

(defmethod profile-request :twitter [env provider token]
  (one/sign-request
   (providers/oauth-v1-consumer env provider)
   {:as :auto
    :coerce :always
    :throw-exceptions false
    :request-method :get
    :url "https://api.twitter.com/1.1/account/verify_credentials.json"
    :query-params
    {"include_email" "true"
     "skip_statuses" "true"}}
   {:token (:oauth-token token)
    :secret (:oauth-token-secret token)}))

(defn request-profile!
  "Request the user from the OAuth `provider` using `token`."
  [env command provider token]
  (let [request (profile-request env provider token)
        {:keys [status body] :as response} (http/request request)]
    (if (= status 200)
      (infl/hyphenate-keys body)
      (throw (ex-info "Can't request OAuth user."
                      {:type ::request-profile-error
                       :provider provider
                       :response response
                       :token token})))))

(defn- default-redirect-url
  "Returns the default redirect URL."
  [env]
  (assoc (:web env) :uri (routes/path :signin)))

(defn- redirect-url [env request]
  (or (-> (state env request) :redirect-url parse-url)
      (default-redirect-url env)))

(defn- authorization-denied-error
  "Returns an authorization denied error."
  [{:keys [description] :as provider}]
  (ex-info (format (str "Can't authenticate with %s. You refused to authorize "
                        "the Burning Swell application.") description)
           {:type ::authorization-denied-error
            :hint "Please authorize the Burning Swell application."
            :provider provider}))

(defn- login-refused-error
  "Returns a login refused error."
  [{:keys [description] :as provider}]
  (ex-info (format "Can't authenticate with %s. You refused to login."
                   description)
           {:type ::login-refused-error
            :hint (format (str "Please login to %s and authorize the "
                               "Burning Swell application.") description)
            :provider provider}))

(defmulti validate-params!
  "Validate the callback parameters from the OAuth provider."
  (fn [env provider request] (-> provider :name keyword)))

(defmethod validate-params! :google [env provider request]
  (validate-state! env request))

(defmethod validate-params! :facebook [env provider request]
  (validate-state! env request)
  (case (-> request :params :error)
    "access_denied"
    (throw (authorization-denied-error provider))
    nil))

(defmethod validate-params! :linkedin [env provider request]
  (validate-state! env request)
  (case (-> request :params :error)
    "user_cancelled_authorize"
    (throw (authorization-denied-error provider))
    "user_cancelled_login"
    (throw (login-refused-error provider))
    nil))

(defmethod validate-params! :twitter [env provider request]
  (when (-> request :params :denied)
    (throw (authorization-denied-error provider))))

;; Save user

(defn save-user!
  "Save the `user`."
  [{:keys [db]} provider user]
  (or (users/update! db provider user)
      (-> (users/insert! db provider user)
          (assoc :created? true))))

(defn- make-command
  "Make the OAuth callback command."
  [env request]
  {:id (core/rand-uuid)
   :name :burningswell.api.commands.oauth/callback
   :provider (provider-name request)
   :params (:params request)})

(defn- publish-success-events!
  "Publish the OAuth callback success event."
  [env command user profile token]
  (when (:created? user)
    (->> {:id (core/rand-uuid)
          :command-id (:id command)
          :name :burningswell.api.events/user-created
          :user-id (:id user)}
         (events/publish! (:publisher env))))
  (->> {:id (core/rand-uuid)
        :command-id (:id command)
        :name :burningswell.api.events.oauth/callback-succeeded
        :user-id (:id user)
        :profile profile
        :token token
        :provider (:provider command)}
       (events/publish! (:publisher env))))

(defn- publish-error-event!
  "Publish the OAuth callback  event."
  [env command error]
  (->> {:id (core/rand-uuid)
        :command-id (:id command)
        :name :burningswell.api.events.oauth/callback-failed
        :error error}
       (events/publish! (:publisher env))))

(defn- provider-callback [env command provider request]
  (log/info {:msg "OAuth callback received."
             :provider (:name provider)
             :request request})
  (validate-params! env provider request)
  (let [token (request-access-token! env command provider request)
        profile (request-profile! env command provider token)
        user (save-user! env provider profile)
        auth-token (jwt/auth-token (:jwt env) user)]
    (publish-success-events! env command user profile token)
    (log/info {:msg "OAuth callback successful."
               :provider (:name provider)
               :request request})
    (-> (redirect-url env request)
        (assoc-in [:query-params :status] "success")
        (assoc-in [:query-params :auth-token] auth-token)
        (format-url)
        (response/redirect))))

(defn- provider-not-found [env command request]
  {:status 404
   :body (format "OAuth provider \"%s\" not found."
                 (provider-name request))})

(defn- error-response
  "Returns a response for the given `error`."
  [env command request description {:keys [hint type]}]
  (publish-error-event! env command type)
  (-> (redirect-url env request)
      (assoc-in [:query-params :description] description)
      (assoc-in [:query-params :error] (some-> type name))
      (assoc-in [:query-params :hint] hint)
      (assoc-in [:query-params :status] "error")
      (format-url)
      (response/redirect)))

(defn callback [env request]
  (let [command (make-command env request)]
    (try+

     (commands/publish! (:publisher env) command)

     (if-let [provider (provider env request)]
       (provider-callback env command provider request)
       (provider-not-found env command request))

     (catch [:type ::authorization-denied-error] data
       (log/warn {:msg (:message &throw-context)})
       (error-response env command request (:message &throw-context) data))

     (catch [:type ::login-refused-error] data
       (log/warn {:msg (:message &throw-context)})
       (error-response env command request (:message &throw-context) data))

     (catch [:type ::invalid-state-error] {:keys [state] :as data}
       (log/error {:msg (:message &throw-context) :state state})
       (error-response env command request (:message &throw-context) data))

     (catch [:type ::request-access-token-error] {:keys [response] :as data}
       (log/error {:msg (:message &throw-context) :response response})
       (error-response env command request (:message &throw-context) data))

     (catch [:type ::request-profile-error] {:keys [response] :as data}
       (log/error {:msg (:message &throw-context) :response response})
       (error-response env command request (:message &throw-context) data))

     (catch Object data
       (log/error (:throwable &throw-context))
       (error-response env command request (:message &throw-context) data)))))
